package ru.yandex.webmaster3.storage.host.moderation.camelcase.dao;

import com.datastax.driver.core.utils.UUIDs;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.DisplayNameRequest;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationRequestState;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

import java.util.UUID;
import java.util.function.Consumer;


/**
 * User: azakharov
 * Date: 25.02.15
 * Time: 17:50
 */
@Repository
public class HostDisplayNameModerationRequestsYDao extends AbstractYDao {
    private static final String TABLE_HOST_DISPLAY_NAME_MODERATION = "host_display_name_moderation";

    protected HostDisplayNameModerationRequestsYDao() {
        super(PREFIX_HOST, TABLE_HOST_DISPLAY_NAME_MODERATION);
    }

    public DisplayNameRequest getDisplayNameRequest(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .order(F.CREATION_DATE.desc())
                .limit(1)
                .queryOne();

    }

    public void saveDisplayNameRequest(DisplayNameRequest request) {
        upsert(
                F.HOST_ID.value(request.getHostId()),
                F.REQUEST_ID.value(request.getRequestId()),
                F.DISPLAY_NAME.value(request.getDisplayName()),
                F.STATE.value(request.getState()),
                F.MODIFICATION_DATE.value(request.getModificationDate()),
                F.CREATION_DATE.value(request.getCreationDate()),
                F.USER_CLOSED.value(false),
                F.USER_ID.value(request.getUserId()),
                F.ASSESSOR_ID.value(request.getAssessorId()))
                .execute();
    }

    public void hideRefused(WebmasterHostId hostId, UUID requestId) {
        update()
                .with(F.USER_CLOSED.set(true))
                .where(F.HOST_ID.eq(hostId))
                .and(F.REQUEST_ID.eq(requestId))
                .and(F.USER_CLOSED.eq(false))
                .execute();
    }

    public void nullifyUserId(long userId) {
        update().with(F.USER_ID.set(null)).where(F.USER_ID.eq(userId)).execute();
    }

    public void forEach(Consumer<DisplayNameRequest> consumer) {
        streamReader(MAPPER, consumer, true);
    }

    private static final class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<UUID> REQUEST_ID = Fields.uuidField("request_id");
        static final Field<HostDisplayNameModerationRequestState> STATE = Fields.intEnumField("state", HostDisplayNameModerationRequestState.R);
        static final Field<String> DISPLAY_NAME = Fields.stringField("display_name");
        static final Field<DateTime> CREATION_DATE = Fields.jodaDateTimeField("creation_date");
        static final Field<DateTime> MODIFICATION_DATE = Fields.jodaDateTimeField("modification_date");
        static final Field<Boolean> USER_CLOSED = Fields.boolField("user_closed");
        static final Field<Long> USER_ID = Fields.longField("user_id").makeOptional();
        static final Field<String> ASSESSOR_ID = Fields.stringField("assessor_id").makeOptional();
    }

    private static final DataMapper<DisplayNameRequest> MAPPER = DataMapper.create(
            F.HOST_ID, F.REQUEST_ID, F.DISPLAY_NAME, F.STATE, F.CREATION_DATE, F.MODIFICATION_DATE,
            F.USER_CLOSED, F.USER_ID, F.ASSESSOR_ID, DisplayNameRequest::new
    );
}
