package ru.yandex.webmaster3.storage.host.moderation.camelcase.dao;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.host.moderation.AbstractHostModerationYtRequest;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationYtRequest;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author leonidrom
 */
@Repository
public class HostDisplayNameModerationYtRequestsYDao extends AbstractYDao {
    private static final String TABLE_NAME = "host_display_name_mod_yt_reqs";

    public HostDisplayNameModerationYtRequestsYDao() {
        super(PREFIX_HOST, TABLE_NAME);
    }

    private static final ValueDataMapper<HostDisplayNameModerationYtRequest> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.REQUEST_ID, AbstractHostModerationYtRequest::getRequestId),
            Pair.of(F.HOST_ID, AbstractHostModerationYtRequest::getHostId),
            Pair.of(F.CURRENT_NAME, HostDisplayNameModerationYtRequest::getCurrentDisplayName),
            Pair.of(F.REQUESTED_NAME, HostDisplayNameModerationYtRequest::getRequestedDisplayName),
            Pair.of(F.USER_COMMENT, HostDisplayNameModerationYtRequest::getUserComment),
            Pair.of(F.CREATED_DATE, AbstractHostModerationYtRequest::getCreatedDate),
            Pair.of(F.AUTO_MODERATED, AbstractHostModerationYtRequest::isAutoModerated)
    );

    public void storeRequest(HostDisplayNameModerationYtRequest entry) {
        storeRequests(Collections.singleton(entry));
    }

    public void storeRequests(Collection<HostDisplayNameModerationYtRequest> requests) {
        batchInsert(INSERT_VALUE_MAPPER, requests).execute();
    }

    public void deleteRequests(List<UUID> batch) {
        batchDelete(DELETE_VALUE_MAPPER, batch).execute();
    }

    public void forEach(Consumer<HostDisplayNameModerationYtRequest> consumer) {
        streamReader(MAPPER, consumer);
    }

    public void forEachRequestId(Consumer<UUID> consumer) {
        streamReader(F.REQUEST_ID, consumer);
    }

    private interface F {
        Field<UUID> REQUEST_ID = Fields.uuidField("request_id");
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<String> CURRENT_NAME = Fields.stringField("current_name");
        Field<String> REQUESTED_NAME = Fields.stringField("requested_name");
        Field<String> USER_COMMENT = Fields.stringField("user_comment").makeOptional();
        Field<DateTime> CREATED_DATE = Fields.jodaDateTimeField("created_date");
        Field<Boolean> AUTO_MODERATED = Fields.boolField("auto_moderated");
    }

    private static final DataMapper<HostDisplayNameModerationYtRequest> MAPPER = DataMapper.create(F.HOST_ID, F.REQUEST_ID,
            F.CREATED_DATE, F.REQUESTED_NAME, F.CURRENT_NAME, F.USER_COMMENT, F.AUTO_MODERATED, HostDisplayNameModerationYtRequest::new);

    private static final ValueDataMapper<UUID> DELETE_VALUE_MAPPER = ValueDataMapper.create2(
        Pair.of(F.REQUEST_ID, Function.identity())
    );
}
