package ru.yandex.webmaster3.storage.host.moderation.camelcase.service;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.host.moderation.AbstractHostModerationYtService;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationResolutionEnum;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationYtRequest;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationYtResult;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.dao.HostDisplayNameModerationYtRequestsYDao;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;

import java.util.*;

/**
 * @author leonidrom
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostDisplayNameModerationYtService extends AbstractHostModerationYtService<HostDisplayNameModerationYtRequest, HostDisplayNameModerationYtResult> {
    private static final String REQUESTS_TABLE_SCHEMA = "[" +
            "{'name': 'RequestId', 'type': 'string'}, " +
            "{'name': 'Host', 'type': 'string'}, " +
            "{'name': 'CurrentName', 'type': 'string'}, " +
            "{'name': 'RequestedName', 'type': 'string'}, " +
            "{'name': 'UserComment', 'type': 'string'}, " +
            "{'name': 'CreatedDate', 'type': 'string'}" +
            "]";

    private static final String RESULTS_TABLE_SCHEMA = "[" +
            "{'name': 'RequestId', 'type': 'string'}, " +
            "{'name': 'Host', 'type': 'string'}, " +
            "{'name': 'CurrentName', 'type': 'string'}, " +
            "{'name': 'RequestedName', 'type': 'string'}, " +
            "{'name': 'UserComment', 'type': 'string'}, " +
            "{'name': 'CreatedDate', 'type': 'string'}, " +
            "{'name': 'ModerationDate', 'type': 'string'}, " +
            "{'name': 'Reject', 'type': 'boolean'}, " +
            "{'name': 'Rules', 'type': 'any'}, " +
            "{'name': 'Comment', 'type': 'string'}, " +
            "{'name': 'Assessor', 'type': 'string'}" +
            "]";

    private final HostDisplayNameModerationYtRequestsYDao hdnModerationYtRequestsYDao;
    private final HostDisplayNameService hostDisplayNameService;

    @Override
    protected void deleteProcessedYtRequests(List<UUID> processedRequests) {
        hdnModerationYtRequestsYDao.deleteRequests(processedRequests);
    }

    @Override
    protected Set<UUID> getProcessedRequestIds() {
        return Collections.emptySet();
    }

    @Override
    protected List<HostDisplayNameModerationYtRequest> listAllRequests() {
        List<HostDisplayNameModerationYtRequest> result = new ArrayList<>();
        hdnModerationYtRequestsYDao.forEach(result::add);
        return result;
    }

    @Override
    protected void writeRequestColumns(TableWriter tw, HostDisplayNameModerationYtRequest request) {
        tw.column(YtRequestFields.REQUEST_ID, request.getRequestId().toString());
        tw.column(YtRequestFields.HOST, IdUtils.hostIdToReadableUrl(request.getHostId()));
        tw.column(YtRequestFields.CURRENT_NAME, request.getCurrentDisplayName());
        tw.column(YtRequestFields.REQUESTED_NAME, request.getRequestedDisplayName());
        tw.column(YtRequestFields.USER_COMMENT, request.getUserComment());
        tw.column(YtRequestFields.CREATED_DATE, request.getCreatedDate().toString());
    }

    @Override
    protected void writeAutoModerationResultColumns(TableWriter tw, HostDisplayNameModerationYtRequest request) {
        tw.column(YtResultFields.MODERATION_DATE, request.getCreatedDate().toString());
        tw.column(YtResultFields.REJECT, false); // здесь авто модерация всегда означет принятую завяку
        tw.column(YtResultFields.ASSESSOR, AUTO_MODERATED_ASSESSOR);
    }

    @Override
    public void processYtResult(HostDisplayNameModerationYtResult row) {
        boolean isRejected = row.isRejected();
        HostDisplayNameModerationResolutionEnum status = isRejected ? HostDisplayNameModerationResolutionEnum.REFUSED : HostDisplayNameModerationResolutionEnum.ACCEPTED;
        hostDisplayNameService.processModerationResult(row.getHostId(), row.getRequestId(), status, row.getAssessor(), row.getModerationDate());
    }

    @Override
    protected Class<HostDisplayNameModerationYtResult> getYtResultClass() {
        return HostDisplayNameModerationYtResult.class;
    }

    @Override
    protected String getRequestsTableSchema() {
        return REQUESTS_TABLE_SCHEMA;
    }

    @Override
    public String getResultsTableSchema() {
        return RESULTS_TABLE_SCHEMA;
    }

    public static class YtRequestFields extends CommonYtRequestFields {
        static final String CURRENT_NAME = "CurrentName";
        static final String REQUESTED_NAME = "RequestedName";
        static final String USER_COMMENT = "UserComment";
    }

    public static class YtResultFields extends CommonYtResultFields {
        static final String REJECT = "Reject";
    }
}
