package ru.yandex.webmaster3.storage.host.moderation.camelcase.service;

import java.util.Collections;
import java.util.List;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.DisplayNameRequest;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationRequestState;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationResolutionEnum;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.HostDisplayNameModerationYtRequest;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.dao.HostDisplayNameModerationRequestsYDao;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.dao.HostDisplayNameModerationYtRequestsYDao;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.dao.HostModeratedDisplayNameYDao;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

/**
 * @author avhaliullin
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostDisplayNameService implements UserTakeoutDataProvider {
    private final DisplayNameService2 displayNameService2;
    private final HostDisplayNameModerationRequestsYDao hdnModerationRequestsYDao;
    private final HostDisplayNameModerationYtRequestsYDao hdnModerationYtRequestsYDao;
    private final HostModeratedDisplayNameYDao hostModeratedDisplayNameYDao;
    private final WMCEventsService wmcEventsService;

    public CancelDisplayNameChangeResult cancelDisplayNameChangeRequest(WebmasterHostId hostId, UUID requestId, long userId) {
        DisplayNameRequest curReq = hdnModerationRequestsYDao.getDisplayNameRequest(hostId);
        if (curReq == null || !requestId.equals(curReq.getRequestId())) {
            return new CancelDisplayNameChangeResult(CancelDisplayNameChangeResultStatus.REQUEST_NOT_FOUND, null);
        } else if (curReq.getState() != HostDisplayNameModerationRequestState.IN_PROGRESS) {
            return new CancelDisplayNameChangeResult(CancelDisplayNameChangeResultStatus.UNABLE_TO_CANCEL, null);
        }

        var now = DateTime.now();
        var newReq = DisplayNameRequest.builder()
                .hostId(curReq.getHostId())
                .requestId(UUIDs.timeBased())
                .displayName(curReq.getDisplayName())
                .state(HostDisplayNameModerationRequestState.CANCELLED)
                .creationDate(now)
                .modificationDate(now)
                .isUserClosedInfoBanner(curReq.isUserClosedInfoBanner())
                .userId(userId)
                .assessorId(null)
                .build();

        hdnModerationRequestsYDao.saveDisplayNameRequest(newReq);
        hdnModerationYtRequestsYDao.deleteRequests(Collections.singletonList(curReq.getRequestId()));

        return new CancelDisplayNameChangeResult(CancelDisplayNameChangeResultStatus.OK, displayNameService2.getDisplayName(hostId));
    }

    public HostDisplayNameChangeResult changeDisplayName(WebmasterHostId hostId, long userId, String newDisplayHostName,
                                                         String userComment, boolean autoModerated) {

        log.info("changeDisplayName: hostId={}, newDisplayName={}, autoModerated={}",
                hostId, newDisplayHostName, autoModerated);
        String currentDisplayName = displayNameService2.getDisplayName(hostId);
        DisplayNameRequest displayNameRequest = hdnModerationRequestsYDao.getDisplayNameRequest(hostId);
        if (displayNameRequest != null) {
            if (displayNameRequest.getState() == HostDisplayNameModerationRequestState.IN_PROGRESS) {
                log.info("changeDisplayName: host {} already has request {} in progress", hostId, displayNameRequest.getRequestId());
                return new HostDisplayNameChangeResult(HostDisplayNameChangeResultStatus.ALREADY_HAS_MODERATION_REQUEST,
                        null, null);
            } else if (displayNameRequest.getState() == HostDisplayNameModerationRequestState.REFUSED) {
                log.info("changeDisplayName: hiding refused request {} for host {}", displayNameRequest.getRequestId(), hostId);
                hdnModerationRequestsYDao.hideRefused(hostId, displayNameRequest.getRequestId());
            }
        }

        DisplayNameRequest pendingDisplayNameRequest = hostModeratedDisplayNameYDao.getLastRequest(hostId);
        if (pendingDisplayNameRequest != null &&
                !pendingDisplayNameRequest.getDisplayName().equals(currentDisplayName) &&
                pendingDisplayNameRequest.getDisplayName().equals(newDisplayHostName)) {
            // there is pending request that doesn't equal to production display name
            // and new display name equals to pending display name
            log.info("changeDisplayName: host {} already has pending moderated request {}", hostId, pendingDisplayNameRequest.getRequestId());

            return new HostDisplayNameChangeResult(HostDisplayNameChangeResultStatus.ALREADY_HAS_PENDING_REQUEST,
                    null, null);
        }

        if ((pendingDisplayNameRequest == null || pendingDisplayNameRequest.getDisplayName().equals(currentDisplayName)) &&
                currentDisplayName.equals(newDisplayHostName)) {
            // there is no pending request or pending request equals to production display name
            // and new display name equals to production display name
            log.info("changeDisplayName: new display name {} for host {} is the same", newDisplayHostName, hostId);

            return new HostDisplayNameChangeResult(HostDisplayNameChangeResultStatus.DISPLAY_NAME_IS_THE_SAME,
                    null, null);
        }

        log.info("changeDisplayName: will create new request for host {}", hostId);

        UUID requestId = UUIDs.timeBased();
        DateTime now = DateTime.now();
        DisplayNameRequest newDisplayNameRequest = DisplayNameRequest.builder()
                .hostId(hostId)
                .requestId(requestId)
                .displayName(newDisplayHostName)
                .state(autoModerated ? HostDisplayNameModerationRequestState.ACCEPTED : HostDisplayNameModerationRequestState.IN_PROGRESS)
                .creationDate(now)
                .modificationDate(now)
                .isUserClosedInfoBanner(false)
                .userId(userId)
                .assessorId(null)
                .build();

        // Insert entry into Yt requests queue. If it is not auto moderated, it will go to Yang
        // If it is auto moderated, it will be added straight to the archive results table
        HostDisplayNameModerationYtRequest entry = new HostDisplayNameModerationYtRequest(hostId, requestId,
                now, newDisplayHostName, currentDisplayName, userComment, autoModerated);
        hdnModerationYtRequestsYDao.storeRequest(entry);
        log.info("changeDisplayName: stored Yt request {} for host {}", requestId, hostId);

        if (autoModerated) {
            hostModeratedDisplayNameYDao.saveRequest(newDisplayNameRequest);
            log.info("changeDisplayName: stored auto moderated request {} for host {}", requestId, hostId);
        }

        hdnModerationRequestsYDao.saveDisplayNameRequest(newDisplayNameRequest);
        log.info("changeDisplayName: stored moderation request {} for host {}", requestId, hostId);

        return new HostDisplayNameChangeResult(HostDisplayNameChangeResultStatus.OK, hostId, newDisplayNameRequest);
    }

    void processModerationResult(WebmasterHostId hostId, UUID requestId, HostDisplayNameModerationResolutionEnum status,
                                 String moderatorId, DateTime moderationDate) {

        log.info("Processing moderation result: {}, {}, {}", requestId, hostId, status);
        DisplayNameRequest request = hdnModerationRequestsYDao.getDisplayNameRequest(hostId);
        if (request == null || !request.getRequestId().equals(requestId)) {
            // Could happen if the request had been deleted after sending it to Yt
            log.info("Request {} not found for host {}", requestId, hostId);
            return;
        }

        if (request.getState() != HostDisplayNameModerationRequestState.IN_PROGRESS) {
            log.info("Invalid state {} for request {}", request.getState(), requestId);
            return;
        }

        final DisplayNameRequest moderatedRequest = DisplayNameRequest.builder()
                .hostId(hostId)
                .requestId(UUIDs.timeBased())
                .displayName(request.getDisplayName())
                .state(status.isAccepted() ? HostDisplayNameModerationRequestState.ACCEPTED : HostDisplayNameModerationRequestState.REFUSED)
                .creationDate(DateTime.now())
                .modificationDate(moderationDate)
                .isUserClosedInfoBanner(false)
                .userId(null)
                .assessorId(moderatorId)
                .build();

        if (status.isAccepted()) {
            hostModeratedDisplayNameYDao.saveRequest(moderatedRequest);
        } else {
            if (request.getUserId() != null) {
                wmcEventsService.addEvent(new UserHostMessageEvent<>(
                        hostId,
                        request.getUserId(),
                        new MessageContent.HostDisplayNameChangeCancelled(
                                hostId
                        ),
                        NotificationType.SITE_DISPLAY_NAME,
                        false,
                        null)
                );
            }
        }

        hdnModerationRequestsYDao.saveDisplayNameRequest(moderatedRequest);
    }

    public HideRefusedResultStatus hideRefusedRequest(WebmasterHostId hostId, UUID requestId) {
        DisplayNameRequest displayNameRequest = hdnModerationRequestsYDao.getDisplayNameRequest(hostId);
        if (displayNameRequest == null || !requestId.equals(displayNameRequest.getRequestId())) {
            return HideRefusedResultStatus.REQUEST_NOT_FOUND;
        } else if (displayNameRequest.getState() != HostDisplayNameModerationRequestState.REFUSED) {
            return HideRefusedResultStatus.UNABLE_TO_HIDE;
        }

        hdnModerationRequestsYDao.hideRefused(hostId, requestId);
        return HideRefusedResultStatus.OK;
    }

    public void hideRefusedRequestIfNecessary(WebmasterHostId hostId) {
        DisplayNameRequest displayNameRequest = hdnModerationRequestsYDao.getDisplayNameRequest(hostId);

        if (displayNameRequest == null || displayNameRequest.getState() != HostDisplayNameModerationRequestState.REFUSED) {
            return;
        }

        hdnModerationRequestsYDao.hideRefused(hostId, displayNameRequest.getRequestId());
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        // просто удалить запись в данном случае нельзя, сломается логика показа запросов,
        // поэтому выставляем user_id в null
        hdnModerationRequestsYDao.nullifyUserId(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                hdnModerationRequestsYDao.getTablePath(),
                hdnModerationYtRequestsYDao.getTablePath()
        );
    }

    public enum HostDisplayNameChangeResultStatus {
        OK,
        DISPLAY_NAME_IS_THE_SAME,
        ALREADY_HAS_MODERATION_REQUEST,
        ALREADY_HAS_PENDING_REQUEST
    }

    public static class HostDisplayNameChangeResult {
        public final HostDisplayNameChangeResultStatus status;
        public final WebmasterHostId webmasterHost;
        public final DisplayNameRequest displayNameRequest;

        public HostDisplayNameChangeResult(HostDisplayNameChangeResultStatus status,
                                           WebmasterHostId webmasterHost,
                                           DisplayNameRequest displayNameRequest) {
            this.status = status;
            this.webmasterHost = webmasterHost;
            this.displayNameRequest = displayNameRequest;
        }
    }

    public enum CancelDisplayNameChangeResultStatus {
        OK,
        REQUEST_NOT_FOUND,
        UNABLE_TO_CANCEL
    }

    public static class CancelDisplayNameChangeResult {
        public final CancelDisplayNameChangeResultStatus status;
        public final String displayName;

        public CancelDisplayNameChangeResult(CancelDisplayNameChangeResultStatus status, String displayName) {
            this.status = status;
            this.displayName = displayName;
        }

    }

    public enum HideRefusedResultStatus {
        OK,
        REQUEST_NOT_FOUND,
        UNABLE_TO_HIDE
    }
}
