package ru.yandex.webmaster3.storage.host.moderation.regions.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.collect.Sets;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.W3RegionInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.regions.RegionUtils;
import ru.yandex.webmaster3.core.regions.W3RegionsTreeService;
import ru.yandex.webmaster3.core.regions.data.HostRegion;
import ru.yandex.webmaster3.core.regions.data.HostRegionSourceTypeEnum;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.dao.HostRegionsCHDao;
import ru.yandex.webmaster3.storage.host.dao.HostRegionsLimitYDao;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostModeratedRegions;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequest;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequestStatus;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationYtRequest;
import ru.yandex.webmaster3.storage.host.moderation.regions.dao.*;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

import static ru.yandex.webmaster3.core.regions.RegionUtils.NON_HIDDEN_REGION_TYPES_PREDICATE;

/**
 * @author leonidrom
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class HostRegionsModerationService {
    private static final Duration YT_REQUEST_TTL = Duration.standardHours(72);
    private static final Logger log = LoggerFactory.getLogger(HostRegionsModerationService.class);

    public static final  Set<Integer> NOT_SPECIFIED_REGION_SET = Collections.singleton(RegionUtils.NOT_SPECIFIED_REGION_ID);

    private final W3RegionsTreeService w3regionsTreeService;
    private final HostRegionsCHDao hostRegionsCHDao;
    private final HostRegionsModerationRequestsYDao hostRegionsModerationRequestsYDao;
    private final HostModeratedRegionsYDao hostModeratedRegionsYDao;
    private final HostRegionsLimitYDao hostRegionsLimitYDao;
    private final WMCEventsService wmcEventsService;
    private final HostRegionsModerationYtRequestsYDao hostRegionsModerationYtRequestsYDao;

    /**
     * Возвращает все регионы хоста для текущего поколения
     */
    public Set<HostRegion> getHostRegionsInfo(WebmasterHostId hostId) {
        return hostRegionsCHDao.getHostRegions(hostId);
    }

    public List<Integer> filterRegions(Collection<HostRegion> regions, HostRegionSourceTypeEnum source,
                                        Predicate<W3RegionInfo> predicate) {
        return regions.stream()
                .filter(region -> region.getSourceType() == source)
                .map(region -> {
                    W3RegionInfo visibleRegion = w3regionsTreeService.getVisibleRegionOrParent(region.getRegionId(), predicate);
                    return visibleRegion == null ? null : visibleRegion.getId();
                })
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());
    }

    /**
     * Возвращает последний запрос на модерацию регионов
     */
    public HostRegionsModerationRequest getModerationRequest(WebmasterHostId hostId) {
        return hostRegionsModerationRequestsYDao.getLastRecord(hostId);
    }

    /**
     * Отменяет заданный запрос на модерацию регионов
     */
    public boolean cancelModerationRequest(WebmasterHostId hostId, UUID requestId) {
        HostRegionsModerationRequest lastRequest = hostRegionsModerationRequestsYDao.getLastRecord(hostId);
        if (lastRequest == null || !lastRequest.getRequestId().equals(requestId) || lastRequest.getStatus().isFinalState()) {
            return false;
        } else {
            hostRegionsModerationRequestsYDao.addRecord(UUIDs.timeBased(), hostId, null,
                    HostRegionsModerationRequestStatus.CANCELLED,
                    lastRequest.getCurrentRegions(), lastRequest.getRequestedRegions(), lastRequest.getAcceptedRegions(),
                    lastRequest.getEvidenceUrl());
            hostRegionsModerationYtRequestsYDao.deleteRequests(List.of(requestId));

            return true;
        }
    }

    /**
     *  Сохраняет в базе отметку, что больше этот запрос пользователю показывать не нужно
     */
    public void markModerationRequestAsHiddenFromUser(WebmasterHostId hostId, UUID requestId) {
        hostRegionsModerationRequestsYDao.hideRecord(hostId, requestId);
    }

    /**
     * Создает новый запрос на модерацию регионов
     */
    public AddRequestStatus createModerationRequest(WebmasterHostId hostId, Set<Integer> requestedRegions,
                                                    String evidenceUrl) {
        return createModerationRequest(hostId, requestedRegions, evidenceUrl, true);
    }

    public AddRequestStatus createModerationRequest(WebmasterHostId hostId, Set<Integer> requestedRegions,
                                                    String evidenceUrl, boolean checkErrors) {
        HostRegionsModerationRequest lastRequest = hostRegionsModerationRequestsYDao.getLastRecord(hostId);
        if (checkErrors && lastRequest != null && !lastRequest.getStatus().isFinalState()) {
            return AddRequestStatus.ALREADY_HAS_REQUEST;
        }

        Set<Integer> currentRegions = getCurrentAndPendingModeratedRegions(hostId).getLeft();
        if (checkErrors && requestedRegions.equals(currentRegions)) {
            return AddRequestStatus.IGNORED;
        }

        Set<Integer> autoModeratedRegions = getAutoModeratedRegions(hostId, requestedRegions);

        // Все что не авто, требует ручной модерации
        Set<Integer> needModerationRegions = Sets.difference(requestedRegions, autoModeratedRegions);
        boolean isFullyAutoModerated = needModerationRegions.isEmpty();

        UUID requestId = UUIDs.timeBased();
        addRequest(requestId, hostId, currentRegions, requestedRegions,
                autoModeratedRegions, needModerationRegions, evidenceUrl);

        if (isFullyAutoModerated) {
            hostModeratedRegionsYDao.setHostRegions(hostId, autoModeratedRegions, currentRegions);
            hostRegionsModerationRequestsYDao.addRecord(requestId, hostId, "auto",
                    HostRegionsModerationRequestStatus.MODERATED,
                    currentRegions, requestedRegions, autoModeratedRegions, evidenceUrl);

            // обычно пользователь оповещается при вычитывании таблицы с результатами модерирования,
            // но автомодерированные регионы там пропускаются, поэтому оповещаем здесь
            wmcEventsService.addEvent(createEventChanged(hostId));
        } else {
            hostRegionsModerationRequestsYDao.addRecord(requestId, hostId, null,
                    HostRegionsModerationRequestStatus.IN_MODERATION,
                    currentRegions, requestedRegions, autoModeratedRegions, evidenceUrl);
        }

        return AddRequestStatus.ADDED;
    }

    public void addRequest(UUID requestId, WebmasterHostId hostId,
                           Set<Integer> currentRegions, Set<Integer> requestedRegions,
                           Set<Integer> autoModeratedRegions, Set<Integer> needModerationRegions,
                           String evidenceUrl) {

        HostRegionsModerationYtRequest request = new HostRegionsModerationYtRequest(requestId, hostId, DateTime.now(),
                currentRegions, requestedRegions, autoModeratedRegions, needModerationRegions, evidenceUrl);

        hostRegionsModerationYtRequestsYDao.storeRequest(request);
    }

    private Set<Integer> getAutoModeratedRegions(WebmasterHostId hostId, Set<Integer> requestedRegions)  {
        Set<HostRegion> hostRegions = hostRegionsCHDao.getHostRegions(hostId);
        Set<Integer> currentTrieRegions = hostRegions.stream()
                .filter(hostRegion -> hostRegion.getSourceType() != HostRegionSourceTypeEnum.CATALOG)
                .map(hostRegion -> w3regionsTreeService.getVisibleRegionOrParent(hostRegion.getRegionId(), RegionUtils.VISIBILITY_PREDICATE))
                .filter(Objects::nonNull)
                .map(W3RegionInfo::getId)
                .collect(Collectors.toSet());

        Set<Integer> autoModeratedRegions = autoModerateRegions(requestedRegions, currentTrieRegions);
        return autoModeratedRegions;
    }

    /**
     * Возвращает пару: текущие отмодерированные регионы и флаг,
     * проросли ли они в поиске
     */
    @Nullable
    private HostModeratedRegions getModeratedRegions(WebmasterHostId hostId) {
        return hostModeratedRegionsYDao.getHostRegions(hostId);
    }

    /**
     *  Обрабатывает результат модерации из Янга
     */
    void processModerationResult(UUID requestId, WebmasterHostId hostId, String assessorId, DateTime moderationDate,
                                 Set<Integer> acceptedRegions) {
        log.info("Processing moderation result: {}, {}, {}", requestId, hostId, acceptedRegions);

            HostRegionsModerationRequest request = hostRegionsModerationRequestsYDao.getLastRecord(hostId);
        if (request == null || !request.getRequestId().equals(requestId)) {
            // Could happen if the request had been deleted after sending it to Yt
            log.info("Request {} not found for host {}", requestId, hostId);
            return;
        }

        if (request.getStatus() != HostRegionsModerationRequestStatus.IN_MODERATION) {
            log.info("Invalid state {} for request {}", request.getStatus(), requestId);
            return;
        }

        acceptedRegions.addAll(request.getAcceptedRegions()); // добавляем автомодерированные

        // Отсекаем случай когда в результате модерации были отклонены все регионы:
        // мы не хотим оставить пользователя совсем без регионов
        if (!acceptedRegions.isEmpty()) {
            Pair<Set<Integer>, Set<Integer>> p = getCurrentAndPendingModeratedRegions(hostId);
            Set<Integer> currentRegions = p.getLeft();

            hostModeratedRegionsYDao.setHostRegions(hostId, acceptedRegions, currentRegions);
        }

        hostRegionsModerationRequestsYDao.addRecord(UUIDs.timeBased(), hostId, assessorId,
                HostRegionsModerationRequestStatus.MODERATED,
                request.getCurrentRegions(), request.getRequestedRegions(), acceptedRegions,
                request.getEvidenceUrl());

        if (acceptedRegions.isEmpty()) {
            wmcEventsService.addEvent(createEventCancelled(hostId));
        } else {
            wmcEventsService.addEvent(createEventChanged(hostId));
        }
    }

    private WMCEvent createEventChanged(WebmasterHostId hostId) {
        return WMCEvent.create(new RetranslateToUsersEvent<>(
                new UserHostMessageEvent<>(
                        hostId,
                        null,
                        new MessageContent.HostRegionChanged(
                                hostId),
                        NotificationType.SITE_REGIONS,
                        false)
        ));
    }

    private WMCEvent createEventCancelled(WebmasterHostId hostId) {
        return WMCEvent.create(new RetranslateToUsersEvent<>(
                new UserHostMessageEvent<>(
                        hostId,
                        null,
                        new MessageContent.HostRegionChangeCancelled(
                                hostId),
                        NotificationType.SITE_REGIONS,
                        false)
        ));
    }

    /**
     * Возвращает пару: текущие (то есть проросшие в поиске) отмодерированные регионы
     * и новые, то есть те, которые должны прорасти в поиске через какое то время
     */
    @NotNull
    public Pair<Set<Integer>, Set<Integer>> getCurrentAndPendingModeratedRegions(WebmasterHostId hostId) {
        HostModeratedRegions hostModeratedRegions = getModeratedRegions(hostId);
        if (hostModeratedRegions == null) {
            return Pair.of(Collections.emptySet(), Collections.emptySet());
        }

        Set<Integer> moderatedRegions = hostModeratedRegions.getRegions();
        Set<HostRegion> regions = hostRegionsCHDao.getHostRegions(hostId);

        Set<Integer> webmasterRegions = new HashSet<>(filterRegions(regions, HostRegionSourceTypeEnum.WEBMASTER,
                NON_HIDDEN_REGION_TYPES_PREDICATE));

        Set<Integer> diff = Sets.difference(moderatedRegions, webmasterRegions);
        if (diff.isEmpty()) {
            return Pair.of(moderatedRegions, Collections.emptySet());
        } else {
            Set<Integer> currentRegions = hostModeratedRegions.getPrevRegions();
            return Pair.of(currentRegions, moderatedRegions);
        }
    }

    public int getHostRegionsLimit(WebmasterHostId hostId) {
        return hostRegionsLimitYDao.getLimit(hostId);
    }

    private Set<Integer> autoModerateRegions(Set<Integer> requestedRegions, Set<Integer> currentRegions) {
        Set<Integer> autoModeratedRegions;
        if (requestedRegions.equals(NOT_SPECIFIED_REGION_SET)) {
            autoModeratedRegions = NOT_SPECIFIED_REGION_SET;
        } else {
            // То, что не изменилось, принимется автоматически
            autoModeratedRegions = Sets.intersection(requestedRegions, currentRegions);
        }

        // TODO: смотреть в историю заявок и автоматически принимать регионы, одобренные ранее

        return autoModeratedRegions;
    }

    public enum AddRequestStatus {
        ADDED,
        IGNORED,
        ALREADY_HAS_REQUEST
    }
}
