package ru.yandex.webmaster3.storage.host.service;

import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.host.dao.CommercialOwnersYDao;
import ru.yandex.webmaster3.storage.util.ydb.YdbYqlService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;

/**
 * @author leonidrom
 */
@Service
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class CommercialOwnersService {
    private static final long CACHE_DURATION_MINUTES = 2;

    public static final String QUERY_PREFIX = "\n" +
            "PRAGMA yt.ForceInferSchema;\n" +
            "$current_timestamp = ${CURRENT_TIMESTAMP}ul;\n" +
            "$update_date = DateTime::FromMilliseconds(cast($current_timestamp as Uint64));\n" +
            YqlFunctions.URL_2_HOST_ID.getFunctionDef() + "\n";

    public static final String DATA_SELECT_QUERY = """
            SELECT\s
               cast($url2HostId(Url::NormalizeWithDefaultHttpScheme(Name)) as Utf8) as host_id,
               cast($update_date as Timestamp) as update_time
            FROM ${SOURCE_TABLE}
            WHERE  QueriesAvgCM2 > 0.3 OR QueriesCommRatioGt04 > 0.3
            """;


    private final YtService ytService;
    @Value("arnold://home/webmaster/prod/analytics/mascot/last_commercial_owners")
    private YtPath ytTablePath;

    private final CommercialOwnersYDao commercialOwnersYDao;
    private final YdbYqlService ydbYqlService;

    private LoadingCache<WebmasterHostId, Boolean> commercialOwnersCache = CacheBuilder.newBuilder()
            .maximumSize(300_000)
            .expireAfterAccess(CACHE_DURATION_MINUTES, TimeUnit.MINUTES)
            .build(new CacheLoader<>() {
                @Override
                public Boolean load(@NotNull WebmasterHostId key) {
                    return commercialOwnersYDao.containsHost(key);
                }
            });

    public boolean isCommercialOwner(WebmasterHostId ownerHostId) {
        return commercialOwnersCache.getUnchecked(ownerHostId);
    }

    public void importYtTableToYdb() {
        Instant now = Instant.now();

        StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                "CURRENT_TIMESTAMP", String.valueOf(now.getMillis()),
                "SOURCE_TABLE", ytTablePath.toYqlPath()
        ));

        ydbYqlService.importToYdb(commercialOwnersYDao.getTablePath(), substitutor.replace(DATA_SELECT_QUERY),
                substitutor.replace(QUERY_PREFIX));
        commercialOwnersYDao.clearOldRecords(now);
    }
}
