package ru.yandex.webmaster3.storage.iks;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.concurrency.AsyncCtx;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.iks.data.Sqi;
import ru.yandex.webmaster3.core.link.DatePoint;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.iks.dao.MdbIksCHDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;

/**
 * Created by ifilippov5 on 16.04.18.
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class IksService {
    private static final int DEFAULT_OWNER_IKS_VALUE = 0;

    private final HostOwnerService hostOwnerService;
    private final IksHistoryService iksHistoryService;
    private final MdbIksCHDao mdbIksCHDao;
    private final SettingsService settingsService;

    public String getMascotOwner(WebmasterHostId hostId) {
        return WwwUtil.cutWww(hostOwnerService.getMascotHostOwner(hostId).getPunycodeHostname());
    }

    public Optional<Sqi> getSqi(WebmasterHostId hostId) {
        return getSqiForOwner(getMascotOwner(hostId));
    }

    public int getIksValue(WebmasterHostId hostId) {
        Optional<Sqi> sqi = getSqi(hostId);
        return sqi.map(Sqi::getIks).orElse(DEFAULT_OWNER_IKS_VALUE);
    }

    public int getIksValueForOwner(String owner) {
        Optional<Sqi> sqi = getSqiForOwner(owner);
        return sqi.map(Sqi::getIks).orElse(DEFAULT_OWNER_IKS_VALUE);
    }

    public Optional<Sqi> getSqiForOwner(String owner) {
        return mdbIksCHDao.getLastIks(owner);
    }

    public Map<WebmasterHostId, Integer> getIksValues(Collection<WebmasterHostId> hostIds) {
        Set<String> owners = hostIds.stream().map(this::getMascotOwner).collect(Collectors.toSet());
        Map<String, Sqi> sqiMap = getSqiForOwners(null, owners);
        return hostIds.stream().map(hostId -> Pair.of(hostId,
                Optional.ofNullable(sqiMap.get(getMascotOwner(hostId))).map(Sqi::getIks).orElse(DEFAULT_OWNER_IKS_VALUE)))
                .collect(W3Collectors.toHashMap());
    }

    public Map<String, Sqi> getSqiForOwners(AsyncCtx asyncCtx, Collection<String> owners) {
        if (owners.isEmpty()) {
            return Collections.emptyMap();
        }
        return mdbIksCHDao.getLastIks(owners);
    }

    public List<DatePoint> getIksHistory(String owner, LocalDate dateFrom, LocalDate dateTo) {
        return getIksHistory(Collections.singleton(owner), dateFrom, dateTo).get(owner);
    }

    /**
     * Получить всю историю с промежуточными точками
     * @param owners
     * @param dateFrom
     * @param dateTo
     * @return
     */
    public Map<String, List<DatePoint>> getIksHistory(Collection<String> owners, LocalDate dateFrom, LocalDate dateTo) {
        Map<String, List<Pair<LocalDate, Long>>> points = mdbIksCHDao.getIksHistory(owners, dateFrom, dateTo);
        Map<String, List<DatePoint>> result = new HashMap<>();
        for (String owner : owners) {
            result.put(owner, iksHistoryService.mapDataToAllDays(points.get(owner), dateTo).stream().map(DatePoint::fromEntry).collect(Collectors.toList()));
        }
        return result;
    }

    public Optional<DateTime> getCurrentIksLastImportDate() {
        CommonDataState state = settingsService.getSettingOrNull(CommonDataType.IKS_LAST_IMPORT_DATE);
        DateTime updateDate = (state == null ? null : DateTime.parse(state.getValue()).withZone(TimeUtils.EUROPE_MOSCOW_ZONE));
        return Optional.ofNullable(updateDate);
    }

    public void updateCurrentIksLastImportDate(DateTime modificationCassandraTableDate) {
        settingsService.update(CommonDataType.IKS_LAST_IMPORT_DATE, String.valueOf(modificationCassandraTableDate.withZone(TimeUtils.EUROPE_MOSCOW_ZONE)));
    }

    public static Pair<Boolean, String> comparePunycodeHostNameWithPunycodeOwnerName(WebmasterHostId hostId, String punycodeOwner) {
        boolean equals = WwwUtil.equalsIgnoreWww(hostId.getPunycodeHostname(), punycodeOwner);
        WebmasterHostId ownerId = IdUtils.urlToHostId(punycodeOwner);
        return Pair.of(equals, ownerId.getReadableHostname());
    }

    public static Sqi defaultSqi(String owner) {
        return new Sqi(owner, "", DEFAULT_OWNER_IKS_VALUE, DEFAULT_OWNER_IKS_VALUE, "");
    }
}
