package ru.yandex.webmaster3.storage.iks.util;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.*;

import javax.annotation.Nullable;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by ifilippov5 on 10.04.18.
 */
public class IksTableUtil {
    private static final Logger log = LoggerFactory.getLogger(IksTableUtil.class);

    public static final String DATABASE = AbstractClickhouseDao.DB_WEBMASTER3;
    public static final String TABLE_NAME_PREFIX = "iks2_";
    public static final String MERGE_TABLE_NAME = TABLE_NAME_PREFIX + "merge";
    private static final Pattern RESULT_TABLE_NAME_PATTERN =
            Pattern.compile(TABLE_NAME_PREFIX + "(\\d+)_merge");

    public static final CHTable TABLE_SPEC = CHTable.builder()
            .database(DATABASE)
            .name(TABLE_NAME_PREFIX + "%s")
            .partitionBy("toYYYYMM(date)")
            .keyField("date", CHPrimitiveType.Date)
            .partNameSuffix("_part_")
            .keyField(F.OWNER, CHPrimitiveType.String)
            .field(F.MAIN_MIRROR, CHPrimitiveType.String)
            .field(F.IKS, CHPrimitiveType.Int64)
            .field(F.PREV_IKS, CHPrimitiveType.Int64)
            .field(F.TITLE, CHPrimitiveType.String)
            .sharded(true)
            .parts(1)
            .build();

    public static class F {
        public static final String OWNER = "owner";
        public static final String IKS = "iks";
        public static final String PREV_IKS = "prev_iks";
        public static final String MAIN_MIRROR = "main_mirror";
        public static final String TITLE = "title";
    }

    public static LocalDate parseDateFromTableName(String tableName) {
        Matcher matcher = RESULT_TABLE_NAME_PATTERN.matcher(tableName);
        if (!matcher.matches() || matcher.groupCount() < 1) {
            return null;
        }
        return new DateTime(Long.valueOf(matcher.group(1))).withZone(TimeUtils.EUROPE_MOSCOW_ZONE).toLocalDate();
    }

    public static Predicate<String> createPredicateByNameAndDate(LocalDate dateFrom, LocalDate dateTo) {
        Function<String, LocalDate> dateParser = IksTableUtil::parseDateFromTableName;
        Predicate<String> tableNamePredicate = name -> true;
        if (dateFrom != null) {
            tableNamePredicate = tableNamePredicate.and(name ->
                    dateParser.apply(name) != null && !dateParser.apply(name).isBefore(dateFrom));
        }
        if (dateTo != null) {
            tableNamePredicate = tableNamePredicate.and(name ->
                    dateParser.apply(name) != null && !dateParser.apply(name).isAfter(dateTo));
        }

        return tableNamePredicate;
    }

    private  static final String UPDATE_TIME_ATTR = "update_time";

    @Nullable
    public static DateTime getIksUpdateTimeOnYT(YtService ytService, YtPath tablePath) throws YtException, InterruptedException {
        return ytService.withoutTransactionQuery(cypressService -> {
            if (!cypressService.exists(tablePath)) {
                log.error("Table {} doesn't exists", tablePath);
                return null;
            }
            YtNode result = cypressService.getNode(tablePath);
            long modificationTimeSeconds = Long.parseLong(result.getNodeMeta().get(UPDATE_TIME_ATTR).asText());
            DateTime dateTime = new Instant(modificationTimeSeconds * 1000L).toDateTime();

            return dateTime.withZone(TimeUtils.EUROPE_MOSCOW_ZONE);
        });
    }

}
