package ru.yandex.webmaster3.storage.importanturls;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.importanturls.dao.ImportantUrlsHistoryCHDao;
import ru.yandex.webmaster3.storage.importanturls.dao.ImportantUrlsLimitYDao;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrl;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlRequest;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlStatus;
import ru.yandex.webmaster3.storage.importanturls.data.RecommendedUrl;
import ru.yandex.webmaster3.storage.searchurl.samples.data.UrlStatusInfo;
import ru.yandex.webmaster3.storage.url.checker2.UrlCheckService;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestData;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestSource;

/**
 * @author avhaliullin
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ImportantUrlsService {
    private static final int RECOMMENDED_URLS_LIMIT = 100;

    private final HostsWithImportantUrlsYDao hostsWithImportantUrlsYDao;
    private final ImportantUrlsYDao importantUrlsYDao;
    private final ImportantUrlsHistoryCHDao mdbImportantUrlsHistoryCHDao;
    private final ImportantUrlsLimitYDao importantUrlsLimitYDao;
    private final RecommendedUrlsCHDao mdbRecommendedUrlsCHDao;
    private final UrlCheckService urlCheckService;

    public List<RecommendedUrl> listRecommendedUrls(WebmasterHostId hostId) {
        List<ImportantUrlRequest> importantUrls = listImportantUrlRequests(hostId);
        return listRecommendedUrls(hostId,
                importantUrls.stream().map(ImportantUrlRequest::getRelativeUrl).collect(Collectors.toSet()));
    }

    public List<RecommendedUrl> listRecommendedUrls(WebmasterHostId hostId, Set<String> importantUrls) {
        List<RecommendedUrl> recommendedUrls = mdbRecommendedUrlsCHDao.listRecommendedUrls(hostId);
        return recommendedUrls.stream()
                .filter(url -> {
                    try {
                        return url.getRelativeUrl().equals(IdUtils.toRelativeUrl(hostId, url.getRelativeUrl(), false));
                    } catch (Exception e) {
                        return false;
                    }
                })
                .peek(url -> {
                    if (importantUrls.contains(url.getRelativeUrl())) {
                        url.setImportantUrl(true);
                    }
                })
                .sorted(Comparator.comparingLong(RecommendedUrl::getClicksCount)
                        .thenComparing(RecommendedUrl::getShowsCount).reversed()
                        .thenComparing(RecommendedUrl::getRelativeUrl))
                .limit(getRecommendedUrlsLimit(hostId))
                .collect(Collectors.toList());
    }

    public List<ImportantUrl> listImportantUrls(WebmasterHostId hostId) {
        List<ImportantUrlRequest> urlRequests = importantUrlsYDao.list(hostId);
        Map<String, ImportantUrl> urlsByPath = mdbImportantUrlsHistoryCHDao.getLastStates(hostId).stream()
                .collect(Collectors.toMap(ImportantUrl::getRelativeUrl, Function.identity(),
                        W3Collectors.replacingMerger()));

        return urlRequests.stream().map(req -> {
            return urlsByPath.getOrDefault(req.getRelativeUrl(), ImportantUrl.createFromRequest(req));
        }).collect(Collectors.toList());
    }

    public void addImportantUrls(List<ImportantUrlRequest> urls) {
        if (!urls.isEmpty()) {
            // если все хорошо, записываем
            List<UrlCheckRequestData> checkUrlRequests = new ArrayList<>();
            for (ImportantUrlRequest url : urls) {
                String urlString = IdUtils.hostIdToUrl(url.getHostId()) + url.getRelativeUrl();
                try {
                    checkUrlRequests.add(new UrlCheckRequestData(
                            url.getHostId(),
                            UUIDs.timeBased(),
                            new URL(urlString),
                            DateTime.now(),
                            UrlCheckRequestSource.IMPORTANT_URL
                    ));
                } catch (MalformedURLException e) {
                    // по идее происходить не должно, потому что к этому моменту все урлы уже должны быть валидированы
                    throw new RuntimeException("Bad url " + urlString, e);
                }
            }
            for (UrlCheckRequestData urlCheckRequestData : checkUrlRequests) {
                urlCheckService.createYtRequest(urlCheckRequestData);
            }
            importantUrlsYDao.add(urls);
            hostsWithImportantUrlsYDao.add(urls);
        }
    }

    public List<ImportantUrlRequest> listImportantUrlRequests(WebmasterHostId hostId) {
        return importantUrlsYDao.list(hostId);
    }

    public int getImportantUrlsCount(WebmasterHostId hostId) {
        return importantUrlsYDao.count(hostId);
    }

    public long getRecommendedUrlsCount(WebmasterHostId hostId) {
        return mdbRecommendedUrlsCHDao.count(hostId);
    }

    public int getSearchableUrlsCount(WebmasterHostId hostId) {
        List<ImportantUrlRequest> urlRequests = importantUrlsYDao.list(hostId);
        Map<String, ImportantUrl> urlsByPath = mdbImportantUrlsHistoryCHDao.getLastStates(hostId)
                .stream()
                .collect(
                        Collectors.toMap(
                                ImportantUrl::getRelativeUrl,
                                Function.identity(),
                                W3Collectors.replacingMerger()
                        )
                );

        int searchPages = 0;

        for (ImportantUrlRequest urlRequest : urlRequests) {
            ImportantUrl url = urlsByPath.get(urlRequest.getRelativeUrl());


            if (Optional.ofNullable(url)
                    .map(ImportantUrl::getLastStatus)
                    .map(ImportantUrlStatus::getSearchInfo)
                    .map(ImportantUrlStatus.SearchInfo::getUrlSearchInfo)
                    .map(UrlStatusInfo::getStatus)
                    .map(SearchUrlStatusEnum::isSearchable)
                    .orElse(false)
            ) {
                searchPages++;
            }
        }

        return searchPages;
    }

    public int getRecommendedUrlsLimit(WebmasterHostId hostId) {
        return importantUrlsLimitYDao.getLimit(hostId).orElse(RECOMMENDED_URLS_LIMIT);
    }

    public ImportantUrlsQuota getImportantUrlsQuota(WebmasterHostId hostId) {
        return new ImportantUrlsQuota(getRecommendedUrlsLimit(hostId), getImportantUrlsCount(hostId));
    }

    public List<ImportantUrlStatus> getHistory(WebmasterHostId hostId, String path) {
        return mdbImportantUrlsHistoryCHDao.getHistory(hostId, path);
    }

    @Getter
    public static class ImportantUrlsQuota {
        private final int quotaTotal;
        private final int quotaUsed;
        private final int quotaRemain;

        public ImportantUrlsQuota(int quotaTotal, int quotaUsed) {
            this.quotaTotal = quotaTotal;
            // есть некоторые хосты, у которых больше урлов чем лимит,
            // поэтому для того чтобы не показывать отрицательное число берем минимум
            this.quotaUsed = Math.min(quotaUsed, quotaTotal);
            quotaRemain = this.quotaTotal - this.quotaUsed;
        }

        public boolean quotaIsEnd() {
            return quotaRemain == 0;
        }
    }
}
