package ru.yandex.webmaster3.storage.importanturls.data;

import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;

import java.util.Comparator;

/**
 * @author Oleg Bazdyrev
 */
@Description("Факт подписки на изменения какой-либо важной страницы")
public class ImportantUrl {
    private final WebmasterHostId hostId;
    private final String relativeUrl;
    private final DateTime addDate;
    @Nullable
    private final ImportantUrlStatus lastStatus;
    private final boolean hasMoreThanOneStatus;

    public ImportantUrl(WebmasterHostId hostId, String relativeUrl, DateTime addDate,
                        ImportantUrlStatus lastStatus, boolean hasMoreThanOneStatus) {
        this.hostId = hostId;
        this.relativeUrl = relativeUrl;
        this.addDate = addDate;
        this.lastStatus = lastStatus;
        this.hasMoreThanOneStatus = hasMoreThanOneStatus;
    }

    @Description("Хост, к которому относится страница")
    public WebmasterHostId getHostId() {
        return hostId;
    }

    @Description("Относительный URL страницы")
    public String getRelativeUrl() {
        return relativeUrl;
    }

    @Description("Дата добавления подписки")
    public DateTime getAddDate() {
        return addDate;
    }

    @Nullable
    @Description("Последний статус страницы, может отсутствовать")
    public ImportantUrlStatus getLastStatus() {
        return lastStatus;
    }

    @Description("Есть ли у страницы более одного статуса в истории")
    public boolean isHasMoreThanOneStatus() {
        return hasMoreThanOneStatus;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ImportantUrl that = (ImportantUrl) o;

        if (!hostId.equals(that.hostId)) return false;
        return relativeUrl.equals(that.relativeUrl);
    }

    @Override
    public int hashCode() {
        int result = hostId.hashCode();
        result = 31 * result + relativeUrl.hashCode();
        return result;
    }

    /**
     * Максимальное количество подписок на комбинацию хост-пользователь
     */
    public static Comparator<ImportantUrl> BY_UPDATE_TIME_AND_RELATIVE_URL = (o1, o2) -> {
        long updateTime1 = o1.getLastStatus() == null ? 0L : o1.getLastStatus().getRealUpdateTime().getMillis();
        long updateTime2 = o2.getLastStatus() == null ? 0L : o2.getLastStatus().getRealUpdateTime().getMillis();
        int result = Long.compare(updateTime1, updateTime2);
        if (result != 0) {
            return result;
        }
        return o1.getRelativeUrl().compareTo(o2.getRelativeUrl());
    };

    public static ImportantUrl createFromRequest(ImportantUrlRequest urlRequest) {
        return new ImportantUrl(urlRequest.getHostId(), urlRequest.getRelativeUrl(), urlRequest.getAddDate(), null, false);
    }
}
