package ru.yandex.webmaster3.storage.importanturls.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.joda.time.DateTime;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlStatus.SearchInfo;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;

import java.util.List;

/**
 * Содержимое сообщения об изменении чего-либо на отслеживаемой страницу
 *
 * @author Oleg Bazdyrev
 */
public abstract class ImportantUrlChange<T> extends MessageContent.HostMessageContent {

    //  на какой странице что изменилось
    protected final String relativeUrl;
    // старое и новое значение
    protected final T oldValue;
    protected final T newValue;

    protected ImportantUrlChange(MessageTypeEnum type, WebmasterHostId hostId, String relativeUrl,
                                 T oldValue, T newValue) {
        super(type, hostId);
        this.relativeUrl = relativeUrl;
        this.oldValue = oldValue;
        this.newValue = newValue;
    }

    public String getUrl() {
        return IdUtils.hostIdToReadableUrl(getHostId()) +
                URLEncodeUtil.prettifyUrl(relativeUrl);
    }

    public String getRelativeUrl() {
        return URLEncodeUtil.prettifyUrl(relativeUrl);
    }

    public static class TitleChange extends ImportantUrlChange<String> {

        @JsonCreator
        public TitleChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldTitle") String oldValue,
                @JsonProperty("newTitle") String newValue) {
            super(MessageTypeEnum.URL_TITLE_CHANGE, hostId, relativeUrl, oldValue, newValue);
        }

        public String getOldTitle() {
            return oldValue;
        }

        public String getNewTitle() {
            return newValue;
        }
    }

    public static class IndexingLastAccessChange extends ImportantUrlChange<DateTime> {

        @JsonCreator
        public IndexingLastAccessChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldLastAccess") DateTime oldValue,
                @JsonProperty("newLastAccess") DateTime newValue) {
            super(MessageTypeEnum.URL_INDEXING_LAST_ACCESS_CHANGE, hostId, relativeUrl, oldValue, newValue);
        }

        public DateTime getOldLastAccess() {
            return oldValue;
        }

        public DateTime getNewLastAccess() {
            return newValue;
        }
    }

    public static class SearchLastAccessChange extends ImportantUrlChange<DateTime> {

        @JsonCreator
        public SearchLastAccessChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldLastAccess") DateTime oldValue,
                @JsonProperty("newLastAccess") DateTime newValue) {
            super(MessageTypeEnum.URL_SEARCH_LAST_ACCESS_CHANGE, hostId, relativeUrl, oldValue, newValue);
        }

        public DateTime getOldLastAccess() {
            return oldValue;
        }

        public DateTime getNewLastAccess() {
            return newValue;
        }
    }

    public static class HttpCodeChange extends ImportantUrlChange<HttpCodeInfo> {

        @JsonCreator
        public HttpCodeChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldHttpCode") HttpCodeInfo oldHttpCode,
                @JsonProperty("newHttpCode") HttpCodeInfo newHttpCode) {
            super(MessageTypeEnum.URL_INDEXING_STATUS_CHANGE, hostId, relativeUrl, oldHttpCode,
                    newHttpCode);
        }

        public HttpCodeInfo getNewHttpCode() {
            return newValue;
        }

        public HttpCodeInfo getOldHttpCode() {
            return oldValue;
        }
    }

    public static class SearchUrlStatusChange extends ImportantUrlChange<SearchInfo> {

        @JsonCreator
        public SearchUrlStatusChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldSearchUrlStatus") SearchInfo oldSearchUrlStatus,
                @JsonProperty("newSearchUrlStatus") SearchInfo newSearchUrlStatus) {
            super(MessageTypeEnum.URL_SEARCH_STATUS_CHANGE, hostId, relativeUrl, oldSearchUrlStatus, newSearchUrlStatus);
        }

        public SearchInfo getOldSearchUrlStatus() {
            return oldValue;
        }

        public SearchInfo getNewSearchUrlStatus() {
            return newValue;
        }
    }

    public static class DescriptionChange extends ImportantUrlChange<String> {
        @JsonCreator
        public DescriptionChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldDescription") String oldDescription,
                @JsonProperty("newDescription") String newDescription) {
            super(MessageTypeEnum.URL_DESCRIPTION_CHANGE, hostId, relativeUrl, oldDescription, newDescription);
        }

        public String getOldDescription() {
            return oldValue;
        }

        public String getNewDescription() {
            return newValue;
        }
    }

    public static class RelCanonicalTargetChange extends ImportantUrlChange<String> {
        @JsonCreator
        public RelCanonicalTargetChange(
                @JsonProperty("hostId") WebmasterHostId hostId,
                @JsonProperty("relativeUrl") String relativeUrl,
                @JsonProperty("oldRelCanonicalTarget") String oldRelCanonicalTarget,
                @JsonProperty("newRelCanonicalTarget") String newRelCanonicalTarget) {
            super(MessageTypeEnum.URL_REL_CANONICAL_TARGET_CHANGE, hostId,
                  relativeUrl, oldRelCanonicalTarget, newRelCanonicalTarget);
        }

        public String getOldRelCanonicalTarget() {
            return oldValue;
        }

        public String getNewRelCanonicalTarget() {
            return newValue;
        }
    }
}
