package ru.yandex.webmaster3.storage.importer.dao;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.importer.model.ImportStage;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableInfo;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableState;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Upsert;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * Created by Oleg Bazdyrev on 25/09/2020.
 **/
@Slf4j
@Repository
public class ClickhouseTablesYDao extends AbstractYDao {

    private static final TypeReference<Map<ImportStage, JsonNode>> DATA_MAP_REFERENCE = new TypeReference<>() {
    };
    static final DataMapper<MdbClickhouseTableInfo> MAPPER = DataMapper.create(F.ID, F.CREATED, F.UPDATED, F.DATA, F.DATABASE,
            F.DISTRIBUTED_TABLE_NAME, F.LOCAL_TABLE_NAME, F.ALL_TABLE_NAMES, F.STATE, F.SHARDS, MdbClickhouseTableInfo::new);
    private static final String TABLE_NAME = "clickhouse_tables";

    public ClickhouseTablesYDao() {
        super(PREFIX_IMPORTER, TABLE_NAME);
    }

    public void update(MdbClickhouseTableInfo tableInfo) {
        Upsert upsert = upsert(
                F.ID.value(tableInfo.getId()),
                F.CREATED.value(tableInfo.getCreated()),
                F.UPDATED.value(tableInfo.getUpdated()),
                F.DATA.value(tableInfo.getData()),
                F.DATABASE.value(tableInfo.getDatabase()),
                F.DISTRIBUTED_TABLE_NAME.value(tableInfo.getDistributedTableName()),
                F.LOCAL_TABLE_NAME.value(tableInfo.getLocalTableName()),
                F.ALL_TABLE_NAMES.value(tableInfo.getAllTableNames()),
                F.STATE.value(tableInfo.getState()),
                F.SHARDS.value(tableInfo.getShards())
        );
        execute(upsert);
    }

    public void delete(MdbClickhouseTableInfo tableInfo) {
        execute(delete().where(F.ID.eq(tableInfo.getId())).and(F.CREATED.eq(tableInfo.getCreated())).getStatement());
    }

    public List<MdbClickhouseTableInfo> listTables() {
        return select(MAPPER).order(F.CREATED.asc()).queryForList();
    }

    public List<MdbClickhouseTableInfo> listTables(String id) {
        return select(MAPPER).where(F.ID.eq(id)).order(F.CREATED.asc()).queryForList();
    }

    public List<MdbClickhouseTableInfo> listTables(Collection<String> id) {
        return select(MAPPER).where(F.ID.in(id)).order(F.CREATED.asc()).queryForList();
    }

    public interface F {
        Field<String> ID = Fields.stringField("id");
        Field<DateTime> CREATED = Fields.jodaDateTimeField("created");
        Field<DateTime> UPDATED = Fields.jodaDateTimeField("updated");
        Field<Map<ImportStage, JsonNode>> DATA = Fields.jsonField2("data", DATA_MAP_REFERENCE);
        Field<String> DATABASE = Fields.stringField("database");
        Field<String> DISTRIBUTED_TABLE_NAME = Fields.stringField("distributed_table_name");
        Field<String> LOCAL_TABLE_NAME = Fields.stringField("local_table_name");
        Field<List<String>> ALL_TABLE_NAMES = Fields.jsonField2("all_table_names", JsonMapping.STRING_LIST_REFERENCE);
        Field<MdbClickhouseTableState> STATE = Fields.stringEnumField("state", EnumResolver.er(MdbClickhouseTableState.class));
        Field<Integer> SHARDS = Fields.intField("shards");
    }

}
