package ru.yandex.webmaster3.storage.importer.model;

import java.lang.reflect.Method;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.protobuf.Descriptors;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.apache.commons.lang3.text.StrLookup;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.jetbrains.annotations.Nullable;
import org.springframework.context.ApplicationContext;
import org.springframework.context.expression.MapAccessor;
import org.springframework.expression.AccessException;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.EvaluationException;
import org.springframework.expression.PropertyAccessor;
import org.springframework.expression.TypedValue;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.DataBindingMethodResolver;
import org.springframework.expression.spel.support.DataBindingPropertyAccessor;
import org.springframework.expression.spel.support.SimpleEvaluationContext;

import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.importer.dao.ClickhouseImportTasksYDao;
import ru.yandex.webmaster3.storage.importer.dao.ClickhouseTablesYDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.transfer.YtTransferManager;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.yql.YqlService;

/**
 * Created by Oleg Bazdyrev on 25/09/2020.
 * All than can be used in completion of stages
 */
@Value
@Builder(toBuilder = true)
public class ImportContext {

    private static final String SUBST_PREFIX = "#{";
    private static final String SUBST_SUFFIX = "}";
    private static final SpelExpressionParser EXPRESSION_PARSER = new SpelExpressionParser();

    ImportDefinition definition;
    ImportTask task;
    ApplicationContext applicationContext;
    MdbClickhouseTableInfo table;
    YtCypressService cypressService;
    YtCypressService locksCypressService;
    YtPath workDir;
    YtPath locksNode;

    public MdbClickhouseServer getClickhouseServer() {
        return applicationContext.getBean(MdbClickhouseServer.class);
    }

    public ClickhouseImportTasksYDao getClickhouseImportTasksYDao() {
        return applicationContext.getBean(ClickhouseImportTasksYDao.class);
    }

    public ClickhouseTablesYDao getClickhouseTablesYDao() {
        return applicationContext.getBean(ClickhouseTablesYDao.class);
    }

    public ClickhouseSystemTablesCHDao getClickhouseSystemTablesCHDao() {
        return applicationContext.getBean("mdbClickhouseSystemTablesCHDao", ClickhouseSystemTablesCHDao.class);
    }

    public YqlService getYqlService() {
        return applicationContext.getBean(YqlService.class);
    }

    public YtTransferManager getYtTransferManager() {
        return applicationContext.getBean(YtTransferManager.class);
    }

    public StrSubstitutor createSubstitutor(@Nullable YqlQueryBuilder yqlQueryBuilder) {
        return new StrSubstitutor(new ContextStrLookup(yqlQueryBuilder), SUBST_PREFIX, SUBST_SUFFIX, '#');
    }

    @AllArgsConstructor
    private final class ContextStrLookup extends StrLookup<String> {

        private final YqlQueryBuilder queryBuilder;

        @Override
        public String lookup(String key) {
            switch (key) {
                case "TRANSACTION_ID":
                    return getCypressService().getTransactionId();
                case "YT_CLUSTER":
                    return getTask().getSourceTable().getCluster();
                case "SOURCE_TABLE":
                    return getTask().getSourceTable().getPathWithoutCluster();
                case "INTERMEDIATE_TABLE":
                    return getTask().getIntermediateTable().getPathWithoutCluster();
                case "URL_2_HOST_ID":
                    queryBuilder.appendFDefinition(YqlFunctions.URL_2_HOST_ID);
                    return "$url2HostId";
                case "CUT_WWW_AND_M":
                    queryBuilder.appendFDefinition(YqlFunctions.CUT_WWW_AND_M);
                    return "$CutWWWM";
            }
            // custom functions (protobufs)
            if (key.startsWith("PARSE_PROTO_")) {
                String[] parts = key.substring("PARSE_PROTO_".length()).split("/");
                if (parts.length != 2) {
                    return null;
                }
                String message = parts[0];
                String className = parts[1];
                try {
                    Class<?> protoClass = Class.forName(className);
                    Method method = protoClass.getMethod("getDescriptor");
                    Descriptors.FileDescriptor fileDescriptor = (Descriptors.FileDescriptor) method.invoke(null);
                    queryBuilder.appendFDefinition(YqlFunctions.protoDefinition(message, YqlFunctions.protoMeta(fileDescriptor)));
                    return "$parse_" + message.replaceAll("\\.", "_");
                } catch (Exception e) {
                    return null;
                }
            }
            // try to evaluate expression
            try {
                EvaluationContext context = SimpleEvaluationContext.forPropertyAccessors(
                        DataBindingPropertyAccessor.forReadOnlyAccess(), new MapAccessor(), new JsonNodePropertyAccessor())
                        .withMethodResolvers(DataBindingMethodResolver.forInstanceMethodInvocation()).build();
                Object value = EXPRESSION_PARSER.parseExpression(key).getValue(context, ImportContext.this);
                return value == null ? null : value.toString();
            } catch (EvaluationException ignored) {
                return null;
            }
        }
    }

    public static final class JsonNodePropertyAccessor implements PropertyAccessor {

        private static final Class<?>[] SPECIFIC_TARGET_CLASSES = {JsonNode.class};

        @Override
        public Class<?>[] getSpecificTargetClasses() {
            return SPECIFIC_TARGET_CLASSES;
        }

        @Override
        public boolean canRead(EvaluationContext context, Object target, String name) throws AccessException {
            return ((JsonNode) target).has(name);
        }

        @Override
        public TypedValue read(EvaluationContext context, Object target, String name) throws AccessException {
            return new TypedValue(((JsonNode) target).get(name));
        }

        @Override
        public boolean canWrite(EvaluationContext context, Object target, String name) throws AccessException {
            return false;
        }

        @Override
        public void write(EvaluationContext context, Object target, String name, Object newValue) throws AccessException {

        }
    }

}
