package ru.yandex.webmaster3.storage.importer.model;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.ToString;
import lombok.Value;
import lombok.experimental.Wither;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

/**
 * Created by Oleg Bazdyrev on 20/09/2020.
 */
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
@Builder(toBuilder = true)
@ToString(exclude = {"definition"})
@Wither
public class ImportTask {

    String id;
    @Builder.Default
    boolean enabled = true;
    ImportStage stage;
    YtPath sourceTable;
    YtPath intermediateTable;
    @Builder.Default
    Map<ImportStage, JsonNode> data = new EnumMap<>(ImportStage.class);
    Map<ImportStage, JsonNode> prevData;
    String database;
    String distributedTableName;
    String localTableName;
    List<String> allTableNames;
    DateTime started;
    DateTime updated;
    DateTime finished;
    String error;
    ImportStage errorStage;
    ImportDefinition definition;

    public static <T> Optional<T> getData(Map<ImportStage, JsonNode> data, ImportStage stage, Class<T> clazz) {
        if (data != null && data.containsKey(stage)) {
            return Optional.of(JsonMapping.readValue(data.get(stage).traverse(), clazz));
        } else {
            return Optional.empty();
        }
    }

    public <T> Optional<T> getData(ImportStage stage, Class<T> clazz) {
        return getData(this.data, stage, clazz);
    }

    public <T> Optional<T> getPrevData(ImportStage stage, Class<T> clazz) {
        return getData(this.prevData, stage, clazz);
    }

    public Map<ImportStage, JsonNode> clearData(ImportStage stage) {
        Map<ImportStage, JsonNode> newData = data.isEmpty() ? new EnumMap<>(ImportStage.class) : new EnumMap<>(data);
        newData.remove(stage);
        return newData;
    }

    public Map<ImportStage, JsonNode> putData(ImportStage stage, Object o) {
        Map<ImportStage, JsonNode> newData = data.isEmpty() ? new EnumMap<>(ImportStage.class) : new EnumMap<>(data);
        newData.put(stage, JsonMapping.OM.valueToTree(o));
        return newData;
    }

    public static ImportTask fromDefinition(ImportDefinition definition) {
        return ImportTask.builder()
                .id(definition.getId())
                .enabled(definition.isEnabled())
                .stage(ImportStage.INIT)
                .database(definition.getDatabase())
                .started(DateTime.now())
                .updated(DateTime.now())
                .definition(definition)
                .build();
    }

    public ImportTask refreshDefinition(ImportDefinition definition) {
        return toBuilder()
                .stage(ImportStage.INIT)
                .database(definition.getDatabase())
                .started(DateTime.now())
                .updated(DateTime.now())
                .definition(definition)
                .data(new EnumMap<>(ImportStage.class))
                .error(null)
                .errorStage(null)
                .build();
    }

    @JsonIgnore
    public ImportStage getNextStage() {
        return ImportStage.values()[stage.ordinal() + 1];
    }

    @JsonIgnore
    public LocalDate getImportedDate() {
        return definition.getInitPolicy().getImportedDate(stage == ImportStage.SWITCH ? data : prevData);
    }

    public ImportTask.ImportTaskBuilder updateStage(ImportStage stage) {
        return toBuilder().stage(stage).updated(DateTime.now());
    }

    public ImportTask.ImportTaskBuilder withNextStage() {
        return updateStage(getNextStage());
    }

    public ImportTask.ImportTaskBuilder updateError(String error) {
        return updateStage(ImportStage.FAILED).error(error).errorStage(stage);
    }

}
