package ru.yandex.webmaster3.storage.importer.model;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.CityHash102;
import ru.yandex.webmaster3.core.util.NumericHash;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

/**
 * Created by Oleg Bazdyrev on 25/09/2020.
 */
@Value
@Builder(toBuilder = true)
@AllArgsConstructor(onConstructor_ = @JsonCreator)
public class MdbClickhouseTableInfo implements LocalClickhouseTableProvider {

    String id; // same id in ImportDefinition
    DateTime created;
    DateTime updated;
    Map<ImportStage, JsonNode> data; // same data from ImportTask
    String database;
    String distributedTableName;
    String localTableName;
    List<String> allTableNames;
    MdbClickhouseTableState state;
    int shards;

    public static final Comparator<MdbClickhouseTableInfo> BY_UPDATED =
            Comparator.comparing(MdbClickhouseTableInfo::getUpdated).thenComparing(MdbClickhouseTableInfo::getCreated);

    public <T> Optional<T> getData(ImportStage stage, Class<T> clazz) {
        return ImportTask.getData(data, stage, clazz);
    }

    public static MdbClickhouseTableInfo.MdbClickhouseTableInfoBuilder fromContext(ImportContext context) {
        return fromTask(context.getTask(), context.getClickhouseServer().getShardsCount());
    }

    public static MdbClickhouseTableInfo.MdbClickhouseTableInfoBuilder fromTask(ImportTask task, int shards) {
        return MdbClickhouseTableInfo.builder().id(task.getId())
                .created(task.getStarted())
                .updated(DateTime.now())
                .data(task.getData())
                .database(task.getDatabase())
                .distributedTableName(task.getDistributedTableName())
                .localTableName(task.getLocalTableName())
                .allTableNames(task.getAllTableNames())
                .state(MdbClickhouseTableState.ONLINE)
                .shards(shards);
    }

    @Override
    public String getTableName(ClickhouseServer clickhouseServer, WebmasterHostId key) {
        return database + "." + localTableName;
    }

    public int getShard(ClickhouseServer clickhouseServer, Object key) {
        long hash = NumericHash.combineHashes(0L, CityHash102.cityHash64(String.valueOf(key)));
        return (int) Long.remainderUnsigned(hash, shards);
    }

    @Override
    public String getDistrTableName() {
        return database + "." + distributedTableName;
    }
}
