package ru.yandex.webmaster3.storage.importer.model.clear;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.Duration;

import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.importer.model.ImportContext;
import ru.yandex.webmaster3.storage.importer.model.ImportTask;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableInfo;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableState;

/**
 * Created by Oleg Bazdyrev on 25/09/2020.
 */
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
@Builder
@Slf4j
@JsonIgnoreProperties(ignoreUnknown = true)
public class ImportClearFixedNumberOfBackups implements ImportClearPolicy {

    private static final Set<MdbClickhouseTableState> STATES_FOR_CLEARING = EnumSet.of(MdbClickhouseTableState.ONLINE, MdbClickhouseTableState.ARCHIVED);

    int count;
    String groupBy;

    @Override
    public ImportTask apply(ImportContext context) {
        Map<String, List<MdbClickhouseTableInfo>> groupedTables = new HashMap<>();
        for (MdbClickhouseTableInfo table : context.getClickhouseTablesYDao().listTables(context.getDefinition().getId())) {
            if (!STATES_FOR_CLEARING.contains(table.getState())) {
                continue;
            }
            String groupValue = "";
            if (!Strings.isNullOrEmpty(groupBy)) {
                groupValue = context.toBuilder().table(table).build().createSubstitutor(null).replace(groupBy);
            }
            groupedTables.computeIfAbsent(groupValue, (k) -> new ArrayList<>()).add(table);
        }
        // sort
        groupedTables.values().forEach(tables -> {
            tables.sort(MdbClickhouseTableInfo.BY_UPDATED);
            while (tables.size() > count) {
                // хитрая логика, может быть несколько табличек с одним именем, реально будем удалять из
                // кликхауса, если это последняя таблица
                MdbClickhouseTableInfo table = tables.get(0);
                boolean removeChTables = tables.stream().filter(cti -> cti.getDistributedTableName().equals(table.getDistributedTableName())).count() == 1;
                if (removeChTables) {
                    removeClickhouseTables(context.getClickhouseServer(), table);
                }
                // все снесли, можно удалить запись из Кассандры
                context.getClickhouseTablesYDao().delete(table);
                tables.remove(0);
            }
        });

        return context.getTask().withNextStage().build();
    }

    private void removeClickhouseTables(ClickhouseServer clickhouseServer, MdbClickhouseTableInfo table) {
        log.info("Cleaning up tables: " + table.getAllTableNames());
        // проходимся по каждому хосту
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext
                    .useDefaults()
                    .setTimeout(Duration.standardMinutes(10))
                    .setHost(host);
            for (String tableName : table.getAllTableNames()) {
                clickhouseServer.execute(ctx,
                        ClickhouseServer.QueryType.INSERT, "DROP TABLE IF EXISTS " + table.getDatabase() + "." + tableName,
                        Optional.empty(), Optional.empty());
            }
        }
    }

    @Override
    public ImportClearType getType() {
        return ImportClearType.FIXED_NUMBER_OF_BACKUPS;
    }
}
