package ru.yandex.webmaster3.storage.importer.model.init;

import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.storage.importer.model.ImportContext;
import ru.yandex.webmaster3.storage.importer.model.ImportStage;
import ru.yandex.webmaster3.storage.importer.model.ImportTask;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

/**
 * Created by Oleg Bazdyrev on 08/10/2020.
 */
@Slf4j
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
@Builder
@JsonIgnoreProperties(ignoreUnknown = true)
public class ImportInitByDateRangeInAttribute implements ImportInitPolicy {

    private static final String DATA_SEPARATOR = "_";
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");

    YtPath path;
    String rangeStartAttributeName;
    String rangeEndAttributeName;
    String attributeValuePattern;
    String dateFormat;

    @Override
    public String getCluster() {
        return path.getCluster();
    }

    @Override
    public ImportTask apply(ImportContext context) {
        ImportTask task = context.getTask();
        // читаем указанные атрибуты и извлекаем из них даты
        DateTimeFormatter dateTimeFormatter = DateTimeFormat.forPattern(dateFormat);
        YtNode node = context.getCypressService().getNode(path);
        Pattern valuePattern = Pattern.compile(attributeValuePattern);
        Matcher minMatcher = valuePattern.matcher(node.getNodeMeta().get(rangeStartAttributeName).asText());
        Matcher maxMatcher = valuePattern.matcher(node.getNodeMeta().get(rangeEndAttributeName).asText());
        if (!minMatcher.matches() || !maxMatcher.matches()) {
            return task.withError("Attribute has bad format");
        }
        LocalDate minDate = dateTimeFormatter.parseLocalDate(minMatcher.group(1));
        LocalDate maxDate = dateTimeFormatter.parseLocalDate(maxMatcher.group(1));

        LocalDate curMaxDate = task.getPrevData(ImportStage.INIT, Data.class).map(Data::getMaxDate).orElse(null);

        if (curMaxDate == null || curMaxDate.isBefore(maxDate)) {
            LocalDate newMinDate = curMaxDate == null ? null : curMaxDate.plusDays(1);
            return task.withNextStage().sourceTable(path)
                    .data(task.putData(ImportStage.INIT, new Data(ObjectUtils.max(minDate, newMinDate), maxDate)))
                    .build();
        }

        // ничего не нашли, сразу завершимся
        log.info("No new data found");
        return task.updateStage(ImportStage.DONE).build();
    }

    @Override
    public void validate() {
        Preconditions.checkState(path != null);
    }

    @Override
    public ImportInitType getType() {
        return ImportInitType.BY_DATE_RANGE_IN_ATTRIBUTE;
    }

    @Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    private static class Data {
        @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd")
        LocalDate minDate;
        @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd")
        LocalDate maxDate;
    }

    @Override
    public LocalDate getImportedDate(Map<ImportStage, JsonNode> data) {
        return ImportTask.getData(data, ImportStage.INIT, Data.class).map(Data::getMaxDate).orElse(null);
    }
}
