package ru.yandex.webmaster3.storage.importer.model.init;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.storage.importer.model.ImportContext;
import ru.yandex.webmaster3.storage.importer.model.ImportStage;
import ru.yandex.webmaster3.storage.importer.model.ImportTask;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

/**
 * Created by Oleg Bazdyrev on 15/10/2020.
 */
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
@Builder
@JsonIgnoreProperties(ignoreUnknown = true)
@Slf4j
public class ImportInitTailAndWeekTables implements ImportInitPolicy {

    private static final String TAIL_TABLE_NAME = "tail";
    private static final String WEEKLY_DIR_NAME = "weekly";
    private static final Pattern WEEKLY_TABLE_PATTERN = Pattern.compile("week_([0-9]{8})_([0-9]{8})");
    private static final String ATTR_LAST_PROCESSED = "last_processed";
    private static final LocalDate TAIL_TABLE_DATE = LocalDate.parse("1970-01-01");
    private static final DateTimeFormatter TABLE_NAME_DATE_FORMATTER = DateTimeFormat.forPattern("yyyyMMdd");

    YtPath sourceDir;

    @Override
    public String getCluster() {
        return sourceDir.getCluster();
    }

    @Override
    public ImportTask apply(ImportContext context) {
        ImportTask task = context.getTask();
        // collecting all current tables
        TreeMap<LocalDate, TableInfo> ytTables = new TreeMap<>();

        YtPath tailTablePath = YtPath.path(sourceDir, TAIL_TABLE_NAME);
        if (context.getCypressService().exists(tailTablePath)) {
            YtNode node = context.getCypressService().getNode(tailTablePath);
            ytTables.put(TAIL_TABLE_DATE, TableInfo.builder()
                    .date(TAIL_TABLE_DATE)
                    .updateDate(getLastProcessed(node))
                    .tail(true)
                    .path(tailTablePath)
                    .build()
            );
        }
        YtPath weekDirPath = YtPath.path(sourceDir, WEEKLY_DIR_NAME);
        for (YtPath table : context.getCypressService().list(weekDirPath)) {
            Matcher matcher = WEEKLY_TABLE_PATTERN.matcher(table.getName());
            if (!matcher.matches()) {
                continue;
            }
            LocalDate date = TABLE_NAME_DATE_FORMATTER.parseLocalDate(matcher.group(2));
            YtNode node = context.getCypressService().getNode(tailTablePath);
            ytTables.put(date, TableInfo.builder()
                    .date(date)
                    .updateDate(getLastProcessed(node))
                    .tail(false)
                    .path(table)
                    .build()
            );
        }
        // compare with exists tablesx
        TreeMap<LocalDate, TableInfo> chTables = task.getPrevData(ImportStage.INIT, Data.class).map(Data::getAllTables).orElse(new TreeMap<>());
        // ищем первую таблицу, которой не хватает
        TableInfo tableForImport = null;
        for (var entry : ytTables.entrySet()) {
            LocalDate date = entry.getKey();
            if (!chTables.containsKey(date) || chTables.get(date).getUpdateDate() < entry.getValue().getUpdateDate()) {
                tableForImport = entry.getValue();
                tableForImport.getReplaceDates().add(date);
                // Tail-таблица заменяет предыдущую tail-таблицу,
                // плюс все недельные таблицы, которые есть локально, но отсутствуют на YT
                if (entry.getValue().isTail()) {
                    chTables.keySet().stream().filter(tableDate -> !ytTables.containsKey(tableDate)).forEach(tableForImport.getReplaceDates()::add);
                    tableForImport.getReplaceDates().forEach(chTables::remove);
                }
                break;
            }
        }
        if (tableForImport != null) {
            log.info("Updating table {}", tableForImport.getPath());
            chTables.put(tableForImport.getDate(), tableForImport);
            return task.withNextStage()
                    .started(DateTime.now())
                    .data(task.putData(ImportStage.INIT, new Data(chTables, tableForImport)))
                    .sourceTable(tableForImport.getPath())
                    .build();
        }

        log.info("Expected table not found, nothing to import");
        return task.updateStage(ImportStage.DONE).build();
    }

    @JsonIgnore
    private static long getLastProcessed(YtNode node) {
        if (node == null || node.getNodeMeta() == null || !node.getNodeMeta().has(ATTR_LAST_PROCESSED)) {
            return 0L;
        }
        return node.getNodeMeta().get(ATTR_LAST_PROCESSED).asLong();
    }

    @Override
    public ImportInitType getType() {
        return ImportInitType.TAIL_AND_WEEK_TABLES;
    }

    @Value
    @Builder
    public final static class TableInfo {
        LocalDate date;
        long updateDate;
        boolean tail;
        YtPath path;
        @Builder.Default
        Set<LocalDate> replaceDates = new HashSet<>();
    }

    // @see IndexingSamplesTableInfo
    @Value
    @Builder
    public final static class Data {
        TreeMap<LocalDate, TableInfo> allTables;
        TableInfo currentTable;
    }
}
