package ru.yandex.webmaster3.storage.importer.model.switching;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.joda.time.Duration;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.storage.importer.model.ImportContext;
import ru.yandex.webmaster3.storage.importer.model.ImportStage;
import ru.yandex.webmaster3.storage.importer.model.ImportTask;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableInfo;
import ru.yandex.webmaster3.storage.importer.model.init.ImportInitTailAndWeekTables.Data;
import ru.yandex.webmaster3.storage.importer.model.init.ImportInitTailAndWeekTables.TableInfo;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

/**
 * Created by Oleg Bazdyrev on 16/10/2020.
 */
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
@Builder
@JsonIgnoreProperties(ignoreUnknown = true)
public class ImportSwitchReplaceTailAndWeek implements ImportSwitchPolicy {

    private static final String PREFIX_NEW = "new_";
    private static final String PREFIX_OLD = "old_";

    @Override
    public ImportTask apply(ImportContext context) {
        ImportTask task = context.getTask();
        // drop distributed table
        context.getClickhouseServer().execute(ClickhouseQueryContext.useDefaults(), ClickhouseServer.QueryType.INSERT,
                "DROP TABLE " + task.getDatabase() + "." + task.getDistributedTableName() + " ON CLUSTER " + context.getClickhouseServer().getClusterId() + ";",
                Optional.empty(), Optional.empty());

        var data = context.getTask().getData(ImportStage.INIT, Data.class).orElseThrow();
        // replace tables
        TableInfo currentTable = data.getCurrentTable();
        // найдем заменяемые таблицы (ищем последнюю таблицу с нужным date
        Map<LocalDate, MdbClickhouseTableInfo> currentTables = context.getClickhouseTablesYDao().listTables(task.getId()).stream()
                .collect(Collectors.toMap(
                        table -> table.getData(ImportStage.INIT, Data.class).map(Data::getCurrentTable).map(TableInfo::getDate).orElse(null),
                        Function.identity(),
                        BinaryOperator.maxBy(MdbClickhouseTableInfo.BY_UPDATED)
                ));
        List<MdbClickhouseTableInfo> newTableInfos = new ArrayList<>();
        StringBuilder replaceBuilder = new StringBuilder(1024);
        replaceBuilder.append("RENAME TABLE ").append(task.getDatabase()).append(".").append(task.getLocalTableName());
        String newTableName = task.getLocalTableName().replaceFirst(PREFIX_NEW, "");
        replaceBuilder.append(" TO ").append(task.getDatabase()).append(".").append(newTableName);
        newTableInfos.add(MdbClickhouseTableInfo.fromContext(context).localTableName(newTableName).build());

        for (LocalDate date : currentTable.getReplaceDates()) {
            MdbClickhouseTableInfo tableInfo = currentTables.get(date);
            if (tableInfo == null) {
                continue;
            }
            replaceBuilder.append(", ").append(task.getDatabase()).append(".").append(tableInfo.getLocalTableName());
            String oldTableName = PREFIX_OLD + tableInfo.getLocalTableName();
            replaceBuilder.append(" TO ").append(task.getDatabase()).append(".").append(oldTableName);
            newTableInfos.add(tableInfo.toBuilder().localTableName(oldTableName).build());
        }
        replaceBuilder.append(" ON CLUSTER ").append(context.getClickhouseServer().getClusterId()).append(";");
        context.getClickhouseServer().execute(ClickhouseQueryContext.useDefaults().setTimeout(Duration.standardMinutes(1L)), replaceBuilder.toString());
        newTableInfos.forEach(context.getClickhouseTablesYDao()::update);

        return task.withNextStage().build();
    }

    @Override
    public ImportSwitchType getType() {
        return ImportSwitchType.REPLACE_TAIL_AND_WEEK;
    }
}
