package ru.yandex.webmaster3.storage.indexing2.history;

import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.indexing2.history.dao.IndexedUrlsCountHistoryCHDao;
import ru.yandex.webmaster3.storage.indexing2.history.dao.IndexingEventsCountHistoryCHDao;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexedUrlsCountHistoryRecord;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexingEventsCountHistoryRecord;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexingEventsHistoryIndicatorType;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexingHistoryData;
import ru.yandex.webmaster3.storage.indexing2.internal.dao.IndexingHistoryTablesYDao;
import ru.yandex.webmaster3.storage.indexing2.internal.data.IndexingHistoryImportType;
import ru.yandex.webmaster3.storage.indexing2.internal.data.IndexingHistoryTableImport;
import ru.yandex.webmaster3.storage.indexing2.internal.data.IndexingHistoryTableImportState;

/**
 * Created by leonidrom on 21/02/2017.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class IndexedUrlsCountHistoryService {
    private static final long CACHE_DURATION_MINUTES = 1L;

    private final LoadingCache<IndexingHistoryImportType, String> DATA_VERSION_CACHE = CacheBuilder.newBuilder()
            .maximumSize(10)
            .expireAfterAccess(CACHE_DURATION_MINUTES, TimeUnit.MINUTES)
            .build(new CacheLoader<>() {
                public String load(IndexingHistoryImportType type) {
                    return getDataVersion(type);
                }
            });

    private final IndexedUrlsCountHistoryCHDao mdbIndexedUrlsCountHistoryCHDao;
    private final IndexingEventsCountHistoryCHDao mdbIndexingEventsCountHistoryCHDao;
    private final IndexingHistoryTablesYDao indexingHistoryTablesYDao;

    public NavigableMap<Integer, IndexingHistoryData> getIndexedUrlsCountHistory(
            WebmasterHostId hostId, long nodeId, Instant fromDate, Instant toDate) {

        List<IndexedUrlsCountHistoryRecord> records = mdbIndexedUrlsCountHistoryCHDao.getRecords(hostId, fromDate, toDate, nodeId);
        TreeMap<Integer, IndexingHistoryData> result = new TreeMap<>();
        for (IndexedUrlsCountHistoryRecord record : records) {
            result.computeIfAbsent(record.getStatus(), ign -> new IndexingHistoryData(new TreeMap<>()))
                    .getTimestamp2Value().put(record.getTimestamp(), record.getValue());
        }
        return result;
    }

    public Map<IndexingEventsHistoryIndicatorType, IndexingHistoryData> getIndexingEventsCountHistory(
            WebmasterHostId hostId, long nodeId, Instant fromDate, Instant toDate) {

        List<IndexingEventsCountHistoryRecord> records = mdbIndexingEventsCountHistoryCHDao.getRecords(hostId, fromDate, toDate, nodeId);
        Map<IndexingEventsHistoryIndicatorType, IndexingHistoryData> result = new EnumMap<>(IndexingEventsHistoryIndicatorType.class);
        for (IndexingEventsCountHistoryRecord record : records) {
            for (IndexingEventsHistoryIndicatorType indicator : IndexingEventsHistoryIndicatorType.values()) {
                Map<Instant, Long> historyMap = result.computeIfAbsent(indicator, ign -> new IndexingHistoryData(new TreeMap<>())).getTimestamp2Value();
                switch (indicator) {
                    case CHANGED:
                        historyMap.put(record.getTimestamp(), record.getCountChanged());
                        break;
                    case NEW:
                        historyMap.put(record.getTimestamp(), record.getCountNew());
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown indexing events history indicator: " + indicator);
                }
            }
        }
        return result;
    }

    public String getEventHistoryDataVersion(WebmasterHostId hostId) {
        return DATA_VERSION_CACHE.getUnchecked(IndexingHistoryImportType.EVENTS_COUNT_HISTORY);
    }

    public String getUrlHistoryDataVersion(WebmasterHostId hostId) {
        return DATA_VERSION_CACHE.getUnchecked(IndexingHistoryImportType.URLS_COUNT_HISTORY);
    }

    private String getDataVersion(IndexingHistoryImportType importType) {
        return indexingHistoryTablesYDao.listImportsOfType(importType)
                .stream()
                .filter(t -> t.getState() == IndexingHistoryTableImportState.DONE)
                .map(IndexingHistoryTableImport::getDataUpdateTime)
                .max(Comparator.naturalOrder())
                .map(Instant::toString)
                .orElse("unknown");
    }
}

