package ru.yandex.webmaster3.storage.indexing2.history.dao;

import java.util.List;
import java.util.function.Function;

import lombok.Setter;
import org.apache.commons.lang3.ObjectUtils;
import org.joda.time.Instant;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.FNVHash;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexedUrlsCountHistoryRecord;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * Created by leonidrom on 21/02/2017.
 */
public class IndexedUrlsCountHistoryCHDao extends AbstractClickhouseDao {
    private static final Instant RESHARD_DATE = new Instant(1520895625000L);

    @Setter
    private String tableName = "indexed_urls_count_history_merged";

    public List<IndexedUrlsCountHistoryRecord> getRecords(WebmasterHostId hostId, Instant dateFrom, Instant dateTo, long node)
            throws ClickhouseException {
        dateFrom = ObjectUtils.min(dateFrom, RESHARD_DATE);
        Where where = QueryBuilder.select(
                IndexedUrlsCountHistoryField.NODE_ID.getName(),
                IndexedUrlsCountHistoryField.TIMESTAMP.getName(),
                IndexedUrlsCountHistoryField.STATUS.getName(),
                IndexedUrlsCountHistoryField.VALUE.getName()
        )
                .from(DB_WEBMASTER3_INDEXING, tableName)
                .where(QueryBuilder.eq(IndexedUrlsCountHistoryField.HOST_ID.getName(), hostId.toString()))
                .and(QueryBuilder.eq(IndexedUrlsCountHistoryField.NODE_ID.getName(), node));

        if (dateFrom != null) {
            where = where.and(QueryBuilder.gte(IndexedUrlsCountHistoryField.TIMESTAMP.getName(), dateFrom.getMillis()));
            where = where.and(QueryBuilder.gte(IndexedUrlsCountHistoryField.DATE.getName(), dateFrom.toDateTime().toLocalDate().minusDays(1)));
        }
        if (dateTo != null) {
            where = where.and(QueryBuilder.lte(IndexedUrlsCountHistoryField.TIMESTAMP.getName(), dateTo.getMillis()));
            where = where.and(QueryBuilder.lte(IndexedUrlsCountHistoryField.DATE.getName(), dateTo.toDateTime().toLocalDate().plusDays(1)));
        }

        Function<CHRow, IndexedUrlsCountHistoryRecord> mapper = row -> new IndexedUrlsCountHistoryRecord(
                hostId,
                row.getLong(IndexedUrlsCountHistoryField.NODE_ID.getName()),
                new Instant(row.getLong(IndexedUrlsCountHistoryField.TIMESTAMP.getName())),
                row.getInt(IndexedUrlsCountHistoryField.STATUS.getName()),
                row.getLong(IndexedUrlsCountHistoryField.VALUE.getName())
        );
        return getClickhouseServer().queryAll(
                ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(shard(hostId))),
                where,
                mapper
        );
    }

    private int shard(WebmasterHostId hostId) {
        int shardsCount = getClickhouseServer().getShardsCount();
        return (int) FNVHash.hash64Mod(hostId.toString(), shardsCount);
    }

}
