package ru.yandex.webmaster3.storage.indexing2.history.dao;

import java.util.List;
import java.util.function.Function;

import lombok.Setter;
import org.apache.commons.lang3.ObjectUtils;
import org.joda.time.Instant;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.FNVHash;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexingEventsCountHistoryRecord;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * @author avhaliullin
 */
public class IndexingEventsCountHistoryCHDao extends AbstractClickhouseDao {

    private static final Instant RESHARD_DATE = new Instant(1520895625000L);

    @Setter
    private String tableName = "indexing_events_count_history_merged";

    public List<IndexingEventsCountHistoryRecord> getRecords(WebmasterHostId hostId, Instant dateFrom, Instant dateTo, long node) throws ClickhouseException {
        dateFrom = ObjectUtils.min(dateFrom, RESHARD_DATE);
        Where where = QueryBuilder.select(
                IndexingEventsCountHistoryField.TIMESTAMP.getName(),
                IndexingEventsCountHistoryField.COUNT_NEW.getName(),
                IndexingEventsCountHistoryField.COUNT_CHANGED.getName()
        )
                .from(DB_WEBMASTER3_INDEXING, tableName)
                .where(QueryBuilder.eq(IndexingEventsCountHistoryField.HOST_ID.getName(), hostId.toString()))
                .and(QueryBuilder.eq(IndexingEventsCountHistoryField.NODE_ID.getName(), node));
        if (dateFrom != null) {
            where = where.and(QueryBuilder.gte(IndexingEventsCountHistoryField.TIMESTAMP.getName(), dateFrom.getMillis()));
        }
        if (dateTo != null) {
            where = where.and(QueryBuilder.lte(IndexingEventsCountHistoryField.TIMESTAMP.getName(), dateTo.getMillis()));
        }

        Function<CHRow, IndexingEventsCountHistoryRecord> mapper = row -> new IndexingEventsCountHistoryRecord(
                hostId,
                node,
                new Instant(row.getLong(IndexingEventsCountHistoryField.TIMESTAMP.getName())),
                row.getLong(IndexingEventsCountHistoryField.COUNT_NEW.getName()),
                row.getLong(IndexingEventsCountHistoryField.COUNT_CHANGED.getName())
        );

        return getClickhouseServer().queryAll(
                ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(shard(hostId))),
                where,
                mapper
        );
    }

    private int shard(WebmasterHostId hostId) {
        int shardsCount = getClickhouseServer().getShardsCount();
        return (int) FNVHash.hash64Mod(hostId.toString(), shardsCount);
    }
}
