package ru.yandex.webmaster3.storage.indexing2.history.dao;

import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import ru.yandex.webmaster3.core.util.TimeUtils;

/**
 * @author avhaliullin
 */
public class IndexingHistoryTablesUtil {
    /* Common */
    public static final String DATABASE_NAME = "webmaster3_indexing";

    private static final DateTimeFormatter TABLE_NAME_DATE_FORMATTER = TimeUtils.DF_YYYYMMDD;
    private static final String TABLE_NAME_DATE_RANGE_FORMAT = "%s_%s";

    private static String replicationNodeSuffix(LocalDate tableDate, Instant dataUpdateTime) {
        return tableDate.toString(TABLE_NAME_DATE_FORMATTER) + "_upd" + dataUpdateTime.getMillis();
    }

    /* Urls history */

    public static final String INDEXED_URLS_COUNT_TABLE_NAME_PREFIX = "indexed_urls_count_history_part_";
    private static final String INDEXED_URLS_COUNT_CREATE_SPEC_TEMPLATE = "( " +
            "   date Date,  " +
            "   host_id String,  " +
            "   node_id Int64,  " +
            "   timestamp Int64,  " +
            "   status Int32,  " +
            "   value Int64" +
            ") ENGINE = ReplicatedMergeTree(" +
            "   '/webmaster3/clickhouse/tables/{shard}/webmaster3_indexing/" + INDEXED_URLS_COUNT_TABLE_NAME_PREFIX + "%s', " +
            "   '{replica}', " +
            "   date, " +
            "   (host_id, node_id, timestamp), " +
            "   8192" +
            ")";

    private static final String MDB_INDEXED_URLS_COUNT_CREATE_SPEC_TEMPLATE = "( " +
            "   date Date,  " +
            "   host_id String,  " +
            "   node_id Int64,  " +
            "   timestamp Int64,  " +
            "   status Int32,  " +
            "   value Int64" +
            ") ENGINE = MergeTree()" +
            "   PARTITION BY cityHash64(host_id) %% 16  " +
            "   ORDER BY (host_id, node_id, date) ";

    public static final String INDEXED_URLS_COUNT_INSERT_SPEC = "(" +
            "   date," +
            "   host_id," +
            "   node_id," +
            "   timestamp," +
            "   status," +
            "   value" +
            ") FORMAT TabSeparated";

    public static String getIndexedUrlsCountTableName(LocalDate fromDate, LocalDate toDate) {
        return INDEXED_URLS_COUNT_TABLE_NAME_PREFIX + getDateRangeSuffix(fromDate, toDate);
    }

    public static String getIndexedUrlsCountUpdTableCreateSpec(LocalDate fromDate, Instant dataUpdateTime) {
        return String.format(INDEXED_URLS_COUNT_CREATE_SPEC_TEMPLATE, replicationNodeSuffix(fromDate, dataUpdateTime));
    }

    public static String getMdbIndexedUrlsCountUpdTableCreateSpec(LocalDate fromDate, Instant dataUpdateTime) {
        return String.format(MDB_INDEXED_URLS_COUNT_CREATE_SPEC_TEMPLATE, replicationNodeSuffix(fromDate, dataUpdateTime));
    }

    public static String getIndexedUrlsCountTableCreateSpec(LocalDate fromDate, LocalDate toDate) {
        return String.format(INDEXED_URLS_COUNT_CREATE_SPEC_TEMPLATE, getDateRangeSuffix(fromDate, toDate));
    }

    /* Events history */

    public static final String INDEXING_EVENTS_COUNT_TABLE_NAME_PREFIX = "indexing_events_count_history_part_";
    public static final String INDEXING_EVENTS_COUNT_CREATE_SPEC_TEMPLATE = "( " +
            "   date Date,  " +
            "   host_id String,  " +
            "   node_id Int64,  " +
            "   timestamp Int64,  " +
            "   count_changed Int64," +
            "   count_new Int64" +
            ") ENGINE = ReplicatedMergeTree(" +
            "   '/webmaster3/clickhouse/tables/{shard}/webmaster3_indexing/" + INDEXING_EVENTS_COUNT_TABLE_NAME_PREFIX + "%s', " +
            "   '{replica}', " +
            "   date, " +
            "   (host_id, node_id, timestamp), " +
            "   8192" +
            ")";

    public static final String MDB_INDEXING_EVENTS_COUNT_CREATE_SPEC_TEMPLATE = "( " +
            "   date Date,  " +
            "   host_id String,  " +
            "   node_id Int64,  " +
            "   timestamp Int64,  " +
            "   count_changed Int64," +
            "   count_new Int64" +
            ") ENGINE = MergeTree()" +
            "   PARTITION BY cityHash64(host_id) %% 16 " +
            "   ORDER BY (host_id, node_id, date)";

    public static final String INDEXING_EVENTS_COUNT_INSERT_SPEC = "(" +
            "   date," +
            "   host_id," +
            "   node_id," +
            "   timestamp," +
            "   count_changed, " +
            "   count_new" +
            ") FORMAT TabSeparated";

    public static String getIndexingEventsCountTableName(LocalDate fromDate, LocalDate toDate) {
        return INDEXING_EVENTS_COUNT_TABLE_NAME_PREFIX + getDateRangeSuffix(fromDate, toDate);
    }

    public static String getIndexingEventsCountUpdTableCreateSpec(LocalDate fromDate, Instant dataUpdateTime) {
        return String.format(INDEXING_EVENTS_COUNT_CREATE_SPEC_TEMPLATE, replicationNodeSuffix(fromDate, dataUpdateTime));
    }

    public static String getMdbIndexingEventsCountUpdTableCreateSpec(LocalDate fromDate, Instant dataUpdateTime) {
        return String.format(MDB_INDEXING_EVENTS_COUNT_CREATE_SPEC_TEMPLATE, replicationNodeSuffix(fromDate, dataUpdateTime));
    }

    public static String getIndexingEventsCountTableCreateSpec(LocalDate fromDate, LocalDate toDate) {
        return String.format(INDEXING_EVENTS_COUNT_CREATE_SPEC_TEMPLATE, getDateRangeSuffix(fromDate, toDate));
    }

    private static String getDateRangeSuffix(LocalDate fromDate, LocalDate toDate) {
        return String.format(TABLE_NAME_DATE_RANGE_FORMAT,
                fromDate.toString(TABLE_NAME_DATE_FORMATTER),
                toDate.toString(TABLE_NAME_DATE_FORMATTER));
    }
}
