package ru.yandex.webmaster3.storage.indexing2.internal.data;

import org.joda.time.Instant;
import org.joda.time.LocalDate;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

import java.util.Comparator;
import java.util.List;
import java.util.UUID;

/**
 * @author avhaliullin
 */
public class IndexingHistoryTableImport {
    private final IndexingHistoryImportType importType;
    private final LocalDate tableDate;
    private final Instant dataUpdateTime;
    private final IndexingHistoryTableImportState state;
    private final List<YtPath> ytPreparedTables;
    private final UUID importManagerTaskId;
    private final String tmpTableName;
    private final UUID replicationManagerTaskId;
    private final String replacedTableNewName;

    public IndexingHistoryTableImport(IndexingHistoryImportType importType, LocalDate tableDate, Instant dataUpdateTime,
                                      IndexingHistoryTableImportState state, List<YtPath> ytPreparedTables,
                                      UUID importManagerTaskId, String tmpTableName, UUID replicationManagerTaskId,
                                      String replacedTableNewName) {
        this.importType = importType;
        this.tableDate = tableDate;
        this.dataUpdateTime = dataUpdateTime;
        this.state = state;
        this.ytPreparedTables = ytPreparedTables;
        this.importManagerTaskId = importManagerTaskId;
        this.tmpTableName = tmpTableName;
        this.replicationManagerTaskId = replicationManagerTaskId;
        this.replacedTableNewName = replacedTableNewName;
    }

    @Override
    public String toString() {
        return "{" +
                "importType=" + importType +
                ", tableDate=" + tableDate +
                ", dataUpdateTime=" + dataUpdateTime +
                ", state=" + state +
                '}';
    }

    public String getTmpTableName() {
        return tmpTableName;
    }

    public IndexingHistoryTableImportState getState() {
        return state;
    }

    public IndexingHistoryTableImport restarted() {
        return create(importType, tableDate, dataUpdateTime);
    }

    public LocalDate getTableDate() {
        return tableDate;
    }

    public Instant getDataUpdateTime() {
        return dataUpdateTime;
    }

    public List<YtPath> getYtPreparedTables() {
        return ytPreparedTables;
    }

    public UUID getImportManagerTaskId() {
        return importManagerTaskId;
    }

    public UUID getReplicationManagerTaskId() {
        return replicationManagerTaskId;
    }

    public String getReplacedTableNewName() {
        return replacedTableNewName;
    }

    public IndexingHistoryImportType getImportType() {
        return importType;
    }

    // update data

    public IndexingHistoryTableImport withPreparedTables(List<YtPath> ytPreparedTables) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport withState(IndexingHistoryTableImportState state) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport failed() {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, IndexingHistoryTableImportState.FAILED, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport withImportManagerTaskId(UUID importManagerTaskId) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport withTmpTable(String tmpTableName) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport withReplicationManagerTaskId(UUID replicationManagerTaskId) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    public IndexingHistoryTableImport withReplacedTableNewName(String replacedTableNewName) {
        return new IndexingHistoryTableImport(importType, tableDate, dataUpdateTime, state, ytPreparedTables, importManagerTaskId, tmpTableName, replicationManagerTaskId, replacedTableNewName);
    }

    // static

    public static IndexingHistoryTableImport create(IndexingHistoryImportType importType, LocalDate tableDate, Instant updateTimeForCurrentTable) {
        return new IndexingHistoryTableImport(importType, tableDate, updateTimeForCurrentTable, IndexingHistoryTableImportState.INIT, null, null, null, null, null);
    }

    public static final Comparator<IndexingHistoryTableImport> BY_TABLE_DATE_AND_UPDATE_TIME = Comparator
            .comparing(IndexingHistoryTableImport::getTableDate)
            .thenComparing(IndexingHistoryTableImport::getDataUpdateTime);
}
