package ru.yandex.webmaster3.storage.links.dao;

import com.yandex.ydb.table.values.PrimitiveValue;
import org.joda.time.DateTime;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.link.BrokenLinkSample;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Select;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

/**
 * @author leonidrom
 *
 * В силу особенностей чтения из Ydb приходится держать несколько таблиц
 * с одинаковым содержимиым, но разным порядком полей в первичном ключе
 */
public abstract class AbstractBrokenLinkSamplesYDao extends AbstractYDao {
    protected AbstractBrokenLinkSamplesYDao(String tableSpace, String tableName) {
        super(tableSpace, tableName);
    }

    public void clearOldRecords(DateTime ts, int limit) {
        var sel = getSelectOldRecordsStatement(ts, limit);
        delete().on(sel).execute();
    }

    public boolean hasOldRecords(DateTime ts) {
        return getSelectOldRecordsStatement(ts, 1).queryOne() != null;
    }

    public List<BrokenLinkSample> getSamplesForDstHost(WebmasterHostId dstHostId) {
        List<BrokenLinkSample> result = new ArrayList<>();
        streamReader(DATA_MAPPER, result::add, false,
                PrimitiveValue.utf8(dstHostId.toStringId()),
                PrimitiveValue.utf8(dstHostId.toStringId()));

        return result;
    }

    public List<BrokenLinkSample> getRecentSamplesForDstHost(WebmasterHostId dstHostId, int limit) {
        return select(DATA_MAPPER)
                .where(F.DST_HOST_ID.eq(dstHostId))
                .order(F.LINK_DATE.desc())
                .limit(limit)
                .queryForList();
    }

    public void streamSamplesForDstHost(WebmasterHostId dstHostId, Consumer<BrokenLinkSample> consumer) {
        streamReader(DATA_MAPPER, consumer, false,
                PrimitiveValue.utf8(dstHostId.toStringId()),
                PrimitiveValue.utf8(dstHostId.toStringId()));
    }

    public long getSamplesCountForDstHost(WebmasterHostId hostId) {
        return select(F.DST_HOST_ID.count()).where(F.DST_HOST_ID.eq(hostId)).queryOne();
    }

    public List<BrokenLinkSample> getSamplesForSrcHost(WebmasterHostId srcHostId) {
        List<BrokenLinkSample> result = new ArrayList<>();
        streamReader(DATA_MAPPER, result::add, false,
                PrimitiveValue.utf8(srcHostId.toStringId()),
                PrimitiveValue.utf8(srcHostId.toStringId()));

        return result;
    }

    public List<BrokenLinkSample> getRecentSamplesForSrcHost(WebmasterHostId dstHostId, int limit) {
        return select(DATA_MAPPER)
                .where(F.SRC_HOST_ID.eq(dstHostId))
                .order(F.LINK_DATE.desc())
                .limit(limit)
                .queryForList();
    }

    public void streamSamplesForSrcHost(WebmasterHostId srcHostId, Consumer<BrokenLinkSample> consumer) {
        streamReader(DATA_MAPPER, consumer, false,
                PrimitiveValue.utf8(srcHostId.toStringId()),
                PrimitiveValue.utf8(srcHostId.toStringId()));
    }

    public long getSamplesCountForSrcHost(WebmasterHostId hostId) {
        return select(F.SRC_HOST_ID.count()).where(F.SRC_HOST_ID.eq(hostId)).queryOne();
    }

    private Select<PKKey> getSelectOldRecordsStatement(DateTime ts, int limit) {
        return select(PK_MAPPER)
                .secondaryIndex("table_ts_index")
                .where(F.TABLE_TS.lt(ts))
                .limit(limit);
    }

    private static final DataMapper<BrokenLinkSample> DATA_MAPPER = DataMapper.create(
        F.SRC_HOST_ID, F.SRC_PATH, F.DST_HOST_ID, F.DST_PATH, F.DST_HTTP_CODE, F.LINK_DATE,
            BrokenLinkSample::new
    );

    private record PKKey(WebmasterHostId srcHostId, String srcUrl, WebmasterHostId dstHostId, String dstUrl) {
    }

    private static final DataMapper<PKKey> PK_MAPPER = DataMapper.create(
            F.SRC_HOST_ID, F.SRC_PATH, F.DST_HOST_ID, F.DST_PATH,
            PKKey::new
    );

    private static class F {
        static final Field<WebmasterHostId> SRC_HOST_ID = Fields.hostIdField("src_host_id");
        static final Field<String> SRC_PATH = Fields.stringField("src_path");
        static final Field<WebmasterHostId> DST_HOST_ID = Fields.hostIdField("dst_host_id");
        static final Field<String> DST_PATH = Fields.stringField("dst_path");
        static final Field<Integer> DST_HTTP_CODE = Fields.intField("dst_http_code");
        static final Field<DateTime> LINK_DATE = Fields.jodaDateTimeField("update_date");
        static final Field<DateTime> TABLE_TS = Fields.jodaDateTimeField("table_ts");
    }
}
