package ru.yandex.webmaster3.storage.links.dao;

import org.joda.time.Instant;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.link.HostLinkSample;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.clickhouse.table.InternalLinkSamplesTable;
import ru.yandex.webmaster3.storage.links.InternalLinksOrder;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author aherman
 */
public class InternalLinkSamplesCHDao extends AbstractClickhouseDao {
    public long countSamples(LocalClickhouseTableProvider table, WebmasterHostId hostId, Collection<Integer> httpCodes)
            throws ClickhouseException
    {
        String condition = String.format("host_id = '%s'", hostId.toStringId());

        if (!httpCodes.isEmpty()) {
            String codes =
                    httpCodes.stream().map(c -> Integer.toString(c)).collect(Collectors.joining(","));
            condition += " AND host_http_code IN (" + codes + ")";
        }

        String q = String.format("SELECT count() as count FROM %s WHERE %s",
                table.getTableName(getClickhouseServer(), hostId), condition);

        return getClickhouseServer().queryOne(table.chContext(getClickhouseServer(), hostId), q,
                r -> r.getLongUnsafe("count")).orElse(0L);
    }

    public List<HostLinkSample> listSamples(LocalClickhouseTableProvider table, WebmasterHostId hostId,
            Collection<Integer> httpCodes,
            InternalLinksOrder orderBy,
            long limitFrom, long limitSize)
            throws ClickhouseException
    {
        String condition = String.format("host_id = '%s'", hostId.toStringId());
        if (!httpCodes.isEmpty()) {
            String codes =
                    httpCodes.stream().map(c -> Integer.toString(c)).collect(Collectors.joining(","));
            condition += " AND host_http_code IN (" + codes + ")";
        }

        String orderByStr = getOrderByStatementFragment(orderBy);
        String q = String.format("SELECT * FROM %s WHERE %s " + orderByStr + " LIMIT %d, %d",
                table.getTableName(getClickhouseServer(), hostId), condition, limitFrom, limitSize);

        return getClickhouseServer().queryAll(table.chContext(getClickhouseServer(), hostId), q, r -> {
            String hostPath = r.getString("host_path");
            int hostHttpCode = r.getInt("host_http_code");
            int hostLastAccess = r.getInt("host_last_access");

            String sourcePath = r.getString("source_path");
            WebmasterHostId sourceHostId = IdUtils.stringToHostId(r.getString("source_host_id"));
            int sourceLastAccess = r.getInt("source_last_access");
            int sourceHostIks = r.getInt("source_iks");

            String text = r.getString("link_text");
            int linkDate = r.getInt("link_date");

            String hostUrl = IdUtils.hostIdToUrl(hostId) + (hostPath != null ? hostPath : "");
            String sourceUrl = IdUtils.hostIdToUrl(sourceHostId) + (sourcePath != null ? sourcePath : "");

            return new HostLinkSample(hostId, hostUrl, hostPath, toDate(hostLastAccess), hostHttpCode,
                    new HttpCodeInfo(hostHttpCode), sourceHostId, sourcePath, sourceUrl,
                    toDate(sourceLastAccess), 200, new HttpCodeInfo(200), sourceHostIks, toDate(linkDate), text, null);
        });
    }

    private String getOrderByStatementFragment(InternalLinksOrder orderBy) {
        switch (orderBy) {
            case LINK_DATE:
                return "ORDER BY link_date DESC, host_path ASC";

            case DATE:
            default:
                return "ORDER BY date DESC, host_path ASC, source_path ASC";
        }
    }

    private static Instant toDate(int unixSeconds) {
        return new Instant(TimeUnit.SECONDS.toMillis(unixSeconds));
    }
}
