package ru.yandex.webmaster3.storage.links.dao;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.link.HostLinkStatistics;
import ru.yandex.webmaster3.core.link.LinkHistoryIndicatorType;
import ru.yandex.webmaster3.core.link.LinksHistoryIndicator;
import ru.yandex.webmaster3.core.util.FNVHash;
import ru.yandex.webmaster3.proto.Links;
import ru.yandex.webmaster3.core.proto.converter.LinksProtoConverter;
import ru.yandex.webmaster3.storage.links.RawHttpCodeHistory;
import ru.yandex.webmaster3.storage.links.RawLinkTldHistory;
import ru.yandex.webmaster3.storage.links.RawSimpleLinkHistory;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.GroupableLimitableOrderable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import static ru.yandex.webmaster3.core.link.LinkHistoryIndicatorType.EXTERNAL_LINKS_HTTP_CODES;
import static ru.yandex.webmaster3.core.link.LinkHistoryIndicatorType.INTERNAL_LINKS_HTTP_CODES;

/**
 * Created by Oleg Bazdyrev on 23/05/2017.
 */
public class LinkStatisticsCHDao extends AbstractClickhouseDao {
    private static final String DISTRIBUTED_TABLE_NAME = "link_statistics_distributed";
    private static final String MERGED_TABLE_NAME = "link_statistics_merge";

    private static final LinkHistoryIndicatorType[] BASE_LINK_HISTORY_INDICATOR_TYPES = {
            LinkHistoryIndicatorType.EXTERNAL_LINK_URLS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_HOSTS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_NEW_URLS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_NEW_HOSTS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_GONE_URLS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_GONE_HOSTS_COUNT,
            LinkHistoryIndicatorType.EXTERNAL_LINK_HOST_SQI_GROUPS,
            LinkHistoryIndicatorType.INTERNAL_LINKS_URL_COUNT
    };
    // взято из  ru.yandex.webmaster3.storage.links.dao.LinksHistoryCDao.save()
    private static final Map<LinkHistoryIndicatorType, Function<Links.LinksInfo, Long>> SIMPLE_INDICATOR_FUNCTIONS =
            ImmutableMap.<LinkHistoryIndicatorType, Function<Links.LinksInfo, Long>>builder()
                    .put(LinkHistoryIndicatorType.INTERNAL_LINKS_URL_COUNT, Links.LinksInfo::getInternalLinksCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_URLS_COUNT, Links.LinksInfo::getExternalLinksCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_HOSTS_COUNT, Links.LinksInfo::getExternalHostsCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_NEW_URLS_COUNT, Links.LinksInfo::getNewExternalLinksCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_NEW_HOSTS_COUNT, Links.LinksInfo::getNewExternalHostsCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_GONE_URLS_COUNT, Links.LinksInfo::getGoneExternalLinksCount)
                    .put(LinkHistoryIndicatorType.EXTERNAL_LINK_GONE_HOSTS_COUNT, Links.LinksInfo::getGoneExternalHostsCount)
                    .build();

    public List<HostLinkStatistics> list(WebmasterHostId hostId, int limit) throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.select(F.DATE, F.HOST_ID, F.DATA)
                .from(DB_WEBMASTER3_LINKS, MERGED_TABLE_NAME)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toStringId()))
                .orderBy(F.DATE, OrderBy.Direction.DESC)
                .limit(limit);

        return getClickhouseServer().queryAll(getContext(hostId), st.toString(), LinkStatisticsCHDao::mapHostLinkStatistics);
    }

    public List<HostLinkStatistics> listAfter(WebmasterHostId hostId, DateTime after, int limit) throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.select(F.DATE, F.HOST_ID, F.DATA)
                .from(DB_WEBMASTER3_LINKS, MERGED_TABLE_NAME)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toStringId()))
                .and(QueryBuilder.gt(F.DATE, toClickhouseDate(after.withZone(DateTimeZone.UTC).toLocalDate())))
                .orderBy(F.DATE, OrderBy.Direction.DESC)
                .limit(limit);

        return getClickhouseServer().queryAll(getContext(hostId), st.toString(), LinkStatisticsCHDao::mapHostLinkStatistics);
    }

    private GroupableLimitableOrderable createListForDateRangeStatement(WebmasterHostId hostId, Instant fromDate, Instant toDate) {
        return createListForDateRangeStatement(hostId, fromDate, toDate, false);
    }

    private GroupableLimitableOrderable createListForDateRangeStatement(
            WebmasterHostId hostId, Instant fromDate, Instant toDate, boolean includeFromDate) {
        var fromDateLocal = fromDate.toDateTime().toLocalDate();
        var toDateLocal = toDate.toDateTime().toLocalDate();
        return QueryBuilder.select(F.DATE, F.HOST_ID, F.DATA)
                .from(DB_WEBMASTER3_LINKS, MERGED_TABLE_NAME)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toStringId()))
                .and(includeFromDate? QueryBuilder.gte(F.DATE, fromDateLocal) : QueryBuilder.gt(F.DATE, fromDateLocal))
                .and(QueryBuilder.lte(F.DATE, toDateLocal))
                .orderBy(F.DATE, OrderBy.Direction.DESC);
    }

    private static HostLinkStatistics mapHostLinkStatistics(CHRow chRow) {
        DateTime date = chRow.getDate(F.DATE);
        UUID linkGenerationUUID = UUIDs.startOf(date.getMillis());
        Links.LinksInfo linksInfo = parseLinksInfo(chRow.getBytes(F.DATA));
        return LinksProtoConverter.convert(chRow.getHostId(F.HOST_ID), linkGenerationUUID, linksInfo,
                date.toInstant(), date.toInstant());
    }

    private static Links.LinksInfo parseLinksInfo(byte[] data) {
        try {
            return Links.LinksInfo.parseFrom(data);
        } catch (InvalidProtocolBufferException e) {
            throw new WebmasterException("Error parsing links statistics from protobuf",
                    new WebmasterErrorResponse.UnableToReadRequestBinaryDataResponse(null, e), e);
        }
    }

    public List<LinksHistoryIndicator> getIndicators(WebmasterHostId hostId) throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.select(F.DATE, F.HOST_ID, F.DATA)
                .from(DB_WEBMASTER3_LINKS, MERGED_TABLE_NAME)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toStringId()))
                .orderBy(F.DATE, OrderBy.Direction.DESC);

        // собираем результат
        List<LinksHistoryIndicator> result = new ArrayList<>();
        getClickhouseServer().queryAll(getContext(hostId), st.toString(), chRow -> {
            Instant instant = chRow.getDate(F.DATE).toInstant();
            Links.LinksInfo linksInfo = parseLinksInfo(chRow.getBytes(F.DATA));
            // логика взята из ru.yandex.webmaster3.storage.links.dao.LinksHistoryIndicatorsCDao.updateIndicators()
            // копируем все базовые индикаторы
            for (LinkHistoryIndicatorType indicatorType : BASE_LINK_HISTORY_INDICATOR_TYPES) {
                result.add(new LinksHistoryIndicator(instant, indicatorType, 0));
            }
            // добавляем HTTP-коды
            linksInfo.getInternalLinksHttpCodesList().stream().map(Links.LinkHttpInfo::getHttpCode).forEach(httpCode ->
                    result.add(new LinksHistoryIndicator(instant, INTERNAL_LINKS_HTTP_CODES, httpCode))
            );
            return null;
        });

        return result;
    }

    public RawSimpleLinkHistory getSimpleHistory(WebmasterHostId hostId, Instant fromDate, Instant toDate)
            throws ClickhouseException {

        GroupableLimitableOrderable st = createListForDateRangeStatement(hostId, fromDate, toDate);

        Map<LinkHistoryIndicatorType, Map<Instant, Long>> simpleIndicators = new HashMap<>();
        getClickhouseServer().queryAll(getContext(hostId), st.toString(), chRow -> {
            Instant instant = chRow.getDate(F.DATE).toInstant();
            Links.LinksInfo linksInfo = parseLinksInfo(chRow.getBytes(F.DATA));
            SIMPLE_INDICATOR_FUNCTIONS.forEach((indicatorType, function) -> {
                simpleIndicators.computeIfAbsent(indicatorType, it -> new HashMap<>()).put(instant, function.apply(linksInfo));
            });
            return null;
        });
        return new RawSimpleLinkHistory(simpleIndicators);
    }

    /*
     * leonidrom
     *
     * NOTE: По неустановленным причинам, этом метод изначально отдавал данные с датой строго больше fromDate.
     * Для нужд API нужно отдавать данные с датой больше или равной fromDate. Чтобы ничего случайно не сломать,
     * был добавлен флажок includeFromDate. Возможно, что на самом деле он совсем не нужен, и можно везде
     * использовать gte, но это надо проверять.
     */
    public RawHttpCodeHistory getHttpCodeHistory(
            WebmasterHostId hostId, Instant fromDate, Instant toDate, boolean includeFromDate,
            LinkHistoryIndicatorType indicatorType) throws ClickhouseException {
        Preconditions.checkArgument(indicatorType == INTERNAL_LINKS_HTTP_CODES || indicatorType == EXTERNAL_LINKS_HTTP_CODES);
        GroupableLimitableOrderable st = createListForDateRangeStatement(hostId, fromDate, toDate, includeFromDate);

        Map<Instant, Map<Integer, Long>> httpCodes = new HashMap<>();

        getClickhouseServer().queryAll(getContext(hostId), st.toString(), chRow -> {
            Instant instant = chRow.getDate(F.DATE).toInstant();
            Links.LinksInfo linksInfo = parseLinksInfo(chRow.getBytes(F.DATA));
            if (indicatorType == INTERNAL_LINKS_HTTP_CODES) {
                httpCodes.put(instant, linksInfo.getInternalLinksHttpCodesList().stream()
                        .collect(Collectors.toMap(Links.LinkHttpInfo::getHttpCode, Links.LinkHttpInfo::getCount)));
            } else {
                httpCodes.put(instant, linksInfo.getExternalLinksHttpCodesList().stream()
                        .collect(Collectors.toMap(Links.LinkHttpInfo::getHttpCode, Links.LinkHttpInfo::getCount)));
            }
            return null;
        });
        return new RawHttpCodeHistory(httpCodes);
    }

    public RawLinkTldHistory getTldHistory(WebmasterHostId hostId, Instant fromDate, Instant toDate)
            throws ClickhouseException {

        GroupableLimitableOrderable st = createListForDateRangeStatement(hostId, fromDate, toDate);

        Map<Instant, Map<String, Long>> tldCount = new HashMap<>();
        getClickhouseServer().queryAll(getContext(hostId), st.toString(), chRow -> {
            Instant instant = chRow.getDate(F.DATE).toInstant();
            Links.LinksInfo linksInfo = parseLinksInfo(chRow.getBytes(F.DATA));
            tldCount.put(instant, linksInfo.getExternalLinksTldCountList().stream()
                    .collect(Collectors.toMap(Links.TldInfo::getTldName, Links.TldInfo::getCount)));
            return null;
        });
        return new RawLinkTldHistory(tldCount);
    }

    private ClickhouseQueryContext.Builder getContext(WebmasterHostId hostId) throws ClickhouseException {
        long shard = FNVHash.hash64Mod(hostId.toString(), 10);
        return withShard((int) shard);
    }

    private interface F {
        String DATE = "date";
        String HOST_ID = "host_id";
        String DATA = "data";
    }
}
