package ru.yandex.webmaster3.storage.logging;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.google.common.primitives.UnsignedLong;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.util.CityHash102;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.SimpleByteArrayOutputStream;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Format;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.GroupableLimitableOrderable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;

/**
 * @author tsyplyaev
 */
@Slf4j
public class TasksLoggingCHDao extends AbstractClickhouseDao {
    private static final String DIST_TABLE = "periodic_tasks_log";
    private static final String SHARD_TABLE = "periodic_tasks_log_shard_1";

    private static final ObjectMapper OM = new ObjectMapper()
            .registerModules(new JodaModule(), new WebmasterJsonModule(false))
            .configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);

    public void addEntries(List<TaskLogEntry> entries)
            throws ClickhouseException, JsonProcessingException {
        Statement st = QueryBuilder.insertInto(DB_WEBMASTER3, SHARD_TABLE).fields(
                Fields.TMPDATE.getName(),
                Fields.DATE.getName(),
                Fields.TASK_TYPE.getName(),
                Fields.EVENT_TYPE.getName(),
                Fields.HOST_NAME.getName(),
                Fields.RUN_ID.getName(),
                Fields.RUN_TIME.getName(),
                Fields.STATE.getName(),
                Fields.ERROR.getName()
        ).format(Format.Type.TAB_SEPARATED);

        Map<Integer, List<TaskLogEntry>> shard2Entries = entries.stream().collect(Collectors.groupingBy(this::getShard));

        Map<Integer, ClickhouseHost> shard2Host = new HashMap<>();
        for (int shard : shard2Entries.keySet()) {
            ClickhouseHost host = getClickhouseServer().pickAliveHostOrFail(shard);
            shard2Host.put(shard, host);
        }
        for (int shard : shard2Entries.keySet()) {
            SimpleByteArrayOutputStream bs = new SimpleByteArrayOutputStream();
            for (TaskLogEntry entry : shard2Entries.get(shard)) {
                bs = packRowValues(bs,
                        toClickhouseDate(entry.getDate()),
                        toClickhouseDateTime(entry.getDate()),
                        entry.getTaskType(),
                        entry.getEventType().value(),
                        entry.getHostName(),
                        entry.getRunId(),
                        entry.getRunTime() != null ? OM.writeValueAsString(entry.getRunTime()) : 0,
                        entry.getState() != null ? OM.writeValueAsString(entry.getState()) : "",
                        entry.getError() != null ? OM.writeValueAsString(entry.getError()) : ""
                );
            }

            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults()
                    .setHost(shard2Host.get(shard));
            getClickhouseServer().insert(ctx, st.toString(), bs.toInputStream());
        }
    }


    private int getShard(TaskLogEntry entry){
        byte[] hostIdBytes = entry.getTaskType().getBytes();
        return UnsignedLong.fromLongBits(CityHash102.cityHash64(hostIdBytes, 0, hostIdBytes.length))
                .mod(UnsignedLong.valueOf(getClickhouseServer().getShardsCount())).intValue();
    }

    public long countEntries(DateTime fromDate, DateTime toDate, TaskLogEntry.EventType eventType, String taskType,
                             String hostName, UUID runId, boolean hideZero) throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.select().countAll().from(DB_WEBMASTER3, DIST_TABLE);
        if (runId != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.RUN_ID.getName(), runId));
        }
        if (eventType != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.EVENT_TYPE.getName(), eventType.value()));
        }
        if (fromDate != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.gte(Fields.DATE.getName(), fromDate));
        }
        if (toDate != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.lte(Fields.DATE.getName(), toDate));
        }
        if (taskType != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.TASK_TYPE.getName(), taskType));
        }
        if (hostName != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.HOST_NAME.getName(), hostName));
        }

        if (hideZero) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.gt(Fields.RUN_TIME.getName(), 1000));
        }

        return queryOne(st, r -> r.getLongUnsafe(0)).orElse(0L);
    }

    public List<String> distinctHostnames() throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.selectDistinct(Fields.HOST_NAME.getName()).from(DB_WEBMASTER3, DIST_TABLE);
        return queryAll(st, r -> r.getString(0));
    }

    public List<String> distinctTaskTypes() throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.selectDistinct(Fields.TASK_TYPE.getName()).from(DB_WEBMASTER3, DIST_TABLE);
        return queryAll(st, r -> r.getString(0));
    }

    public List<TaskLogEntry> listLatestEntries(TaskLogEntry.EventType eventType) throws ClickhouseException {
        String aliasPrefix = "agr";
        String query = "SELECT " +
                Fields.TASK_TYPE.name + " AS " + aliasPrefix + Fields.TASK_TYPE.name + ", " +
                "max(" + Fields.DATE.name + ") AS " + aliasPrefix + Fields.DATE.name + ", " +
                argMax(Fields.EVENT_TYPE, Fields.DATE, aliasPrefix) + ", " +
                argMax(Fields.HOST_NAME, Fields.DATE, aliasPrefix) + ", " +
                argMax(Fields.RUN_ID, Fields.DATE, aliasPrefix) + ", " +
                argMax(Fields.RUN_TIME, Fields.DATE, aliasPrefix) + ", " +
                argMax(Fields.STATE, Fields.DATE, aliasPrefix) + ", " +
                argMax(Fields.ERROR, Fields.DATE, aliasPrefix) + " " +
                " FROM " + DB_WEBMASTER3 + "." + DIST_TABLE +
                " WHERE " + Fields.EVENT_TYPE.name + " = " + eventType.value() +
                " GROUP BY " + Fields.TASK_TYPE.name + ";";
        return queryAll(query, getMapper(aliasPrefix));
    }

    private static String argMax(Fields field, Fields by, String aliasPrefix) {
        return "argMax(" + field.name + ", " + by.name + ") AS " + aliasPrefix + field.name;
    }

    public List<TaskLogEntry> listEntries(DateTime fromDate, DateTime toDate, TaskLogEntry.EventType eventType,
                                          String taskType, String hostName, UUID runId, boolean hideZero,
                                          Integer skip, Integer limit,
                                          Fields orderBy, OrderBy.Direction orderDirection) throws ClickhouseException {
        GroupableLimitableOrderable st = QueryBuilder.select().from(DB_WEBMASTER3, DIST_TABLE);
        if (runId != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.RUN_ID.getName(), runId));
        }
        if (eventType != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.EVENT_TYPE.getName(), eventType.value()));
        }
        if (fromDate != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.gte(Fields.DATE.getName(), fromDate));
        }
        if (toDate != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.lte(Fields.DATE.getName(), toDate));
        }
        if (taskType != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.TASK_TYPE.getName(), taskType));
        }
        if (hostName != null) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.eq(Fields.HOST_NAME.getName(), hostName));
        }
        if (hideZero) {
            st = QueryBuilder.applyAndCase(st, QueryBuilder.gt(Fields.RUN_TIME.getName(), 1000));
        }

        if (orderBy != null) {
            st = st.orderBy(orderBy.getName(), orderDirection != null ? orderDirection : OrderBy.Direction.ASC);
        } else {
            st = st.orderBy(Fields.DATE.getName(), OrderBy.Direction.DESC);
        }

        if (limit != null) {
            st = st.limit(skip, limit);
        }

        return queryAll(st, MAPPER);
    }

    public enum Fields {
        TMPDATE("tmpdate"),
        DATE("date"),
        TASK_TYPE("task_type"),
        EVENT_TYPE("event_type"),
        HOST_NAME("host_name"),
        RUN_ID("run_id"),
        RUN_TIME("run_time"),
        STATE("state"),
        ERROR("error"),;

        private final String name;

        Fields(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    private static Function<CHRow, TaskLogEntry> getMapper(String fieldNamePrefix) {
        return r -> {
            UUID rid = r.getStringUUID(fieldNamePrefix + Fields.RUN_ID.getName());
            JsonNode state = null;
            try {
                String c = r.getString(fieldNamePrefix + Fields.STATE.name);
                if (!c.isEmpty()) {
                    state = OM.readTree(c);
                }
            } catch (IOException ex) {
                log.error("JSON error while parsing 'state', runId = {}", rid, ex);
            }

            JsonNode error = null;
            try {
                String c = r.getString(fieldNamePrefix + Fields.ERROR.name);
                if (!c.isEmpty()) {
                    error = OM.readTree(c);
                }
            } catch (IOException ex) {
                log.error("JSON error while parsing 'error', runId = {}", rid, ex);
            }

            return new TaskLogEntry(
                    r.getDateTime(fieldNamePrefix + Fields.DATE.getName()),
                    r.getString(fieldNamePrefix + Fields.TASK_TYPE.getName()),
                    TaskLogEntry.EventType.R.fromValueOrNull(r.getInt(fieldNamePrefix + Fields.EVENT_TYPE.getName())),
                    r.getString(fieldNamePrefix + Fields.HOST_NAME.getName()),
                    rid,
                    r.getLong(fieldNamePrefix + Fields.RUN_TIME.getName()),
                    state,
                    error);
        };
    }

    private static Function<CHRow, TaskLogEntry> MAPPER = getMapper("");
}
