package ru.yandex.webmaster3.storage.logging;

import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.Setter;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;

/**
 * @author tsyplyaev
 */
@Setter
public class TasksLoggingService {
    private TasksLoggingCHDao mdbPeriodicTasksLoggingCHDao;
    private TasksLoggingBatchWriter tasksLoggingBatchWriter;
    private String hostName;

    public void logTaskStart(DateTime date, String taskType, UUID runId) {
        addEntry(date, taskType, TaskLogEntry.EventType.START, hostName, runId, null, null, null, null);
    }

    public void logTaskFinish(DateTime date, String taskType, TaskResult taskResult, UUID runId, long runTime,
                              Object state, Object error, Object ydbTrace) {
        TaskLogEntry.EventType eventType = TaskLogEntry.EventType.FINISH;
        if (taskResult == TaskResult.FAIL) {
            eventType = TaskLogEntry.EventType.ERROR;
        }
        addEntry(date, taskType, eventType, hostName, runId, runTime, state, error, ydbTrace);
    }

    public void addEntry(DateTime date, String taskType, TaskLogEntry.EventType eventType, String hostName,
                         UUID runId, Long runTime, Object state, Object error, Object ydbTrace) {
        try {
            tasksLoggingBatchWriter.addEntry(date, taskType, eventType, hostName, runId, runTime, state, error, ydbTrace);
        } catch (JsonProcessingException ex) {
            throw new WebmasterException("Unable add task log entry",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(this.getClass(), null, ex), ex);
        }
    }

    public long countEntries(DateTime fromDate, DateTime toDate, TaskLogEntry.EventType eventType, String taskType,
                             String hostName, UUID runId, boolean hideZero) {
        try {
            return mdbPeriodicTasksLoggingCHDao.countEntries(fromDate, toDate, eventType, taskType, hostName, runId
                    , hideZero);
        } catch (ClickhouseException ex) {
            throw new WebmasterException("Unable to count entries",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(this.getClass(), null, ex), ex);
        }
    }

    public List<String> distinctHostnames() {
        try {
            return mdbPeriodicTasksLoggingCHDao.distinctHostnames();
        } catch (ClickhouseException ex) {
            throw new WebmasterException("Unable to distinct hostnames",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(this.getClass(), null, ex), ex);
        }
    }

    public List<String> distinctTaskTypes() {
        try {
            return mdbPeriodicTasksLoggingCHDao.distinctTaskTypes();
        } catch (ClickhouseException ex) {
            throw new WebmasterException("Unable to distinct task types",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(this.getClass(), null, ex), ex);
        }
    }

    public List<TaskLogEntry> listEntries(DateTime fromDate, DateTime toDate, TaskLogEntry.EventType eventType,
                                          String taskType, String hostName, UUID runId, boolean hideZero,
                                          Integer skip, Integer limit,
                                          TasksLoggingCHDao.Fields orderBy, OrderBy.Direction orderDirection) {
        try {
            return mdbPeriodicTasksLoggingCHDao.listEntries(fromDate, toDate, eventType, taskType, hostName, runId,
                    hideZero, skip, limit, orderBy, orderDirection);
        } catch (ClickhouseException ex) {
            throw new WebmasterException("Unable to get entries",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(this.getClass(), null, ex), ex);
        }
    }
}
