package ru.yandex.webmaster3.storage.mdb;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.json.JsonMapping;

/**
 * Created by Oleg Bazdyrev on 23/09/2020.
 */
@Slf4j
@Service
public class MdbAuthService extends AbstractExternalAPIService {

    private static final int SOCKET_TIMEOUT = 500;
    private static final String ERROR_MESSAGE = "Error occured while resolving iam token";
    private static final String RESOLVE_TOKEN_METHOD = "tokens";

    private volatile DateTime expiresAt = null;
    private volatile String iamToken = null;

    @Value("${webmaster3.storage.cloud.passport.oauthToken}")
    private String passportOAuthToken;
    @Value("${webmaster3.storage.cloud.iam.apiUrl}")
    private String iamApiUrl;

    private CloseableHttpClient httpClient;

    public void init() throws Exception {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(SOCKET_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();
    }

    public String resolveCloudToken() {
        synchronized (this) {
            if (expiresAt == null || DateTime.now().isAfter(expiresAt)) {
                AuthResponse response = resolveCloudTokenInternal();
                expiresAt = response.expiresAt;
                iamToken = response.iamToken;
            }
            return iamToken;
        }
    }

    @ExternalDependencyMethod("resolveIamToken")
    private AuthResponse resolveCloudTokenInternal() {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpPost post = new HttpPost(iamApiUrl + RESOLVE_TOKEN_METHOD);
            post.setEntity(new StringEntity(JsonMapping.writeValueAsString(new AuthRequest(passportOAuthToken)), ContentType.APPLICATION_JSON));

            try (CloseableHttpResponse httpResponse = httpClient.execute(post)) {
                int code = httpResponse.getStatusLine().getStatusCode();
                String content = EntityUtils.toString(httpResponse.getEntity());
                if (code != HttpStatus.SC_OK) {
                    log.error("Error occured while resolving iam token: {} - {}", code, content);
                    throw new WebmasterException(ERROR_MESSAGE, new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), ERROR_MESSAGE));
                }
                return JsonMapping.readValue(content, AuthResponse.class);
            } catch (IOException e) {
                throw new WebmasterException(ERROR_MESSAGE, new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), ERROR_MESSAGE), e);
            }
        });
    }

    @lombok.Value
    public static final class AuthRequest {
        String yandexPassportOauthToken;
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @Autowired)
    public static final class AuthResponse {
        DateTime expiresAt;
        String iamToken;
    }
}
