package ru.yandex.webmaster3.storage.mdb;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.mdb.model.MdbClickhouseHost;

/**
 * Created by Oleg Bazdyrev on 23/09/2020.
 * https://doc.yandex-team.ru/cloud/managed-clickhouse/api-ref/index.html
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class MdbClickhouseInfoService extends AbstractExternalAPIService {

    private static final int SOCKET_TIMEOUT = 5000;
    private static final String LIST_HOSTS_METHOD = "clusters/%s/hosts";

    private final MdbAuthService mdbAuthService;

    @Value("${webmaster3.storage.mdb.clickhouse.apiUrl}")
    private String mdbApiUrl;

    private CloseableHttpClient httpClient;

    public void init() throws Exception {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(SOCKET_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();
    }

    /**
     * https://doc.yandex-team.ru/cloud/managed-clickhouse/api-ref/Cluster/listHosts.html
     * @param clusterId
     * @return
     */
    @ExternalDependencyMethod("listHosts")
    public List<MdbClickhouseHost> listHosts(String clusterId) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpGet get = new HttpGet(mdbApiUrl + String.format(LIST_HOSTS_METHOD, clusterId));
            String iamToken = mdbAuthService.resolveCloudToken();
            get.addHeader(HttpHeaders.AUTHORIZATION, "Bearer " + iamToken);

            try (CloseableHttpResponse httpResponse = httpClient.execute(get)) {
                int code = httpResponse.getStatusLine().getStatusCode();
                String content = EntityUtils.toString(httpResponse.getEntity());
                if (code != HttpStatus.SC_OK) {
                    log.error("Error occured while listing clickhouse hosts: {} - {}", code, content);
                    throw new WebmasterException("MDB API error", new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "MDB API error"));
                }
                return JsonMapping.readValue(content, ListHostsResponse.class).getHosts();
            } catch (IOException e) {
                throw new WebmasterException("MDB API error", new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "MDB API error"), e);
            }
        });
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static final class ListHostsResponse {
        List<MdbClickhouseHost> hosts;
    }

}
