package ru.yandex.webmaster3.storage.metrika;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.google.common.base.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.metrika.counters.CounterInfo;
import ru.yandex.webmaster3.core.metrika.counters.SimpleCounterInfo;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Created by ifilippov5 on 27.11.17.
 */
public class MetrikaCountersApiService extends AbstractExternalAPIService {
    private static final Logger log = LoggerFactory.getLogger(MetrikaCountersApiService.class);

    private static final String COUNTER_INFO_PATH = "/management/v1/counter/";
    private static final String COUNTERS_LIST_PATH = "/management/v1/counters";

    private static final int SOCKET_TIMEOUT = 10_000;
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new WebmasterJsonModule(false))
            .registerModule(new JodaModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);

    private CloseableHttpClient httpClient;

    private String apiMetrikaCountersUrl;
    private TVMTokenService tvmTokenService;

    public void init() {

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(10, TimeUnit.MINUTES)
                .build();
    }

    /*
     * Забирает через API Метрики данные про счетчик.
     */
    @ExternalDependencyMethod("get-counter-info")
    @NotNull
    public CounterInfo getCounterInfo(long counterId) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            CounterInfo counterInfo;
            HttpGet get = new HttpGet(apiMetrikaCountersUrl + COUNTER_INFO_PATH + counterId);
            get.addHeader(TVMTokenService.TVM2_TICKET_HEADER, tvmTokenService.getToken());
            log.info("Querying metrika {}", get);
            try (CloseableHttpResponse response = httpClient.execute(get)) {
                int httpCode = response.getStatusLine().getStatusCode();
                if (httpCode / 100 == 2) {
                    JsonNode root = OM.readTree(response.getEntity().getContent());
                    log.info("Metrika response {}", root);
                    counterInfo = parseCounterInfo(root);
                    if (counterInfo == null) {
                        throw new WebmasterException(
                                "Unable to parse Metrika response",
                                new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), null);
                    }
                } else if (httpCode / 100 == 4) {
                    log.info("Metrika responded with code {} and content: {}", httpCode, EntityUtils.toString(response.getEntity()));
                    counterInfo = new CounterInfo(counterId, "", "", CounterInfo.CounterStatusEnum.DELETED, "");
                } else {
                    String responseString = IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
                    log.error("Error from Metrika service (" + httpCode + "): " + responseString);
                    throw new WebmasterException(
                            "Error from remote service (" + httpCode + "): " + responseString,
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), null);
                }
            } catch (IOException e) {
                throw new WebmasterException("Unable to query " + get,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
            }

            return counterInfo;
        });
    }

    /*
     * Забирает через API Метрики данные про счетчики пользователя.
     */
    @ExternalDependencyMethod("get-counters-list")
    @NotNull
    public List<SimpleCounterInfo> getCountersList(String userLogin) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpGet get = new HttpGet(apiMetrikaCountersUrl + COUNTERS_LIST_PATH + "?ulogin=" + userLogin);
            get.addHeader(TVMTokenService.TVM2_TICKET_HEADER, tvmTokenService.getToken());
            log.info("Querying metrika {}", get);
            try (CloseableHttpResponse response = httpClient.execute(get)) {
                int httpCode = response.getStatusLine().getStatusCode();
                if (httpCode / 100 == 2) {
                    JsonNode result = OM.readTree(response.getEntity().getContent());
                    //log.info("Metrika response {}", result);
                    ArrayNode counters = (ArrayNode) result.get("counters");
                    List<SimpleCounterInfo> counterInfos = new ArrayList<>();
                    counters.elements().forEachRemaining(n -> counterInfos.add(parseSimpleCounterInfo(n)));
                    return counterInfos;
                } else {
                    String responseString = IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
                    log.error("Error from Metrika service (" + httpCode + "): " + responseString);
                    throw new WebmasterException(
                            "Error from remote service (" + httpCode + "): " + responseString,
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), null);
                }
            } catch (IOException e) {
                throw new WebmasterException("Unable to query " + get,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
            }
        });
    }

    @Nullable
    private CounterInfo parseCounterInfo(JsonNode root) {
        JsonNode counter = root.get("counter");
        if (counter == null) {
            log.error("Missing required 'counter' field in Metrika response");
            return null;
        }

        Long counterId = Optional.ofNullable(counter.get("id")).map(JsonNode::asLong).orElse(null);
        if (counterId == null) {
            log.error("Missing required 'id' field in Metrika response");
            return null;
        }

        Optional<JsonNode> site2 = Optional.ofNullable(counter.get("site2"));
        String site = "";
        if (site2.isPresent()) {
            site = Optional.ofNullable(site2.get().get("site")).map(JsonNode::asText).orElse("");
        }

        String name = Optional.ofNullable(counter.get("name")).map(JsonNode::asText).orElse("");
        String status = Optional.ofNullable(counter.get("status")).map(JsonNode::asText).orElse("deleted");
        CounterInfo.CounterStatusEnum statusEnum = CounterInfo.CounterStatusEnum.valueOf(status.toUpperCase());
        String ownerLogin = Optional.ofNullable(counter.get("owner_login")).map(JsonNode::asText).orElse("");

        return new CounterInfo(counterId, name, site, statusEnum, ownerLogin);
    }

    @Nullable
    private SimpleCounterInfo parseSimpleCounterInfo(JsonNode counter) {
        return SimpleCounterInfo.builder()
                .id(counter.get("id").asLong())
                .type(Optional.ofNullable(counter.get("type")).map(JsonNode::asText).orElse(""))
                .site(Optional.ofNullable(counter.get("site")).map(JsonNode::asText).orElse(""))
                .build();
    }

    @Required
    public void setApiMetrikaCountersUrl(String apiMetrikaCountersUrl) {
        this.apiMetrikaCountersUrl = apiMetrikaCountersUrl;
    }

    @Required
    public void setTvmTokenService(TVMTokenService tvmTokenService) {
        this.tvmTokenService = tvmTokenService;
    }
}
