package ru.yandex.webmaster3.storage.metrika.dao;

import java.util.Base64;
import java.util.List;

import lombok.Setter;
import org.jetbrains.annotations.NotNull;

import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaCrawlSample;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Limit;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.cases.Case;

/**
 * @author leonidrom
 */
public class MetrikaCrawlSamplesCHDao extends AbstractClickhouseDao {
    public static final String TABLE_NAME_PREFIX = "metrika_crawl_samples_";

    @Setter
    private TableProvider tableStorage;

    public static final CHTable TABLE = CHTable.builder()
            .database(AbstractClickhouseDao.DB_WEBMASTER3)
            .name(TABLE_NAME_PREFIX + "%s")
            .partitionBy("toYYYYMM(" + F.DATE + ")")
            .keyField(F.DATE, CHPrimitiveType.Date)
            .keyField(F.DOMAIN, CHPrimitiveType.String)
            .keyField(F.COUNTER_ID, CHPrimitiveType.UInt64)
            .field(F.URL, CHPrimitiveType.String)
            .field(F.TITLE, CHPrimitiveType.String)
            .field(F.TITLE_BASE64, CHPrimitiveType.String)
            .build();

    public long getSamplesCount(String domain, long counterId, Condition condition) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.METRIKA_CRAWL_SAMPLES);

        Where st = QueryBuilder.select()
                .countAll()
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.DOMAIN, domain))
                .and(QueryBuilder.eq(F.COUNTER_ID, counterId));

        st = filter(st, condition);

        return getClickhouseServer().queryOne(
                table.chContext(getClickhouseServer(), domain),
                st.toString(),
                r -> r.getLongUnsafe(0)
        ).orElse(0L);
    }

    public List<MetrikaCrawlSample> getSamples(String domain, long counterId, Condition condition, long limitFrom, long limitSize) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.METRIKA_CRAWL_SAMPLES);

        Where st = QueryBuilder.select(F.URL, F.TITLE_BASE64)
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.DOMAIN, domain))
                .and(QueryBuilder.eq(F.COUNTER_ID, counterId));

        st = filter(st, condition);
        Limit fst = st.limit((int) limitFrom, (int) limitSize);

        List<MetrikaCrawlSample> samples = getClickhouseServer().queryAll(
                table.chContext(getClickhouseServer(), domain),
                fst.toString(),
                chRow ->
                        new MetrikaCrawlSample(
                                domain,
                                counterId,
                                chRow.getString(F.URL),
                                new String(Base64.getDecoder().decode(chRow.getString(F.TITLE_BASE64)))
                        ));

        return samples;
    }

    @NotNull
    public List<Long> getCounterIds(String domain) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.METRIKA_CRAWL_SAMPLES);

        Where st = QueryBuilder.selectDistinct(F.COUNTER_ID)
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.DOMAIN, domain));

        return getClickhouseServer().queryAll(
                table.chContext(getClickhouseServer(), domain),
                st.toString(),
                chRow -> chRow.getLongUnsafe(F.COUNTER_ID));
    }

    public static class F {
        public static final String DATE = "date";
        public static final String DOMAIN = "domain";
        public static final String COUNTER_ID = "counter_id";
        public static final String URL = "url";
        public static final String TITLE = "title";
        public static final String TITLE_BASE64 = "title_base64";
    }

    private Where filter(Where st, Condition filters) {
        if (filters == null) {
            return st;
        }
        String filterStr = filters.toQuery();
        return st.and(new Case() {
            @Override
            public String toString() {
                return filterStr;
            }
        });
    }
}
