package ru.yandex.webmaster3.storage.metrika.dao;

import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaCounterCrawlStateEnum;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @author leonidrom
 */
@Repository
public class MetrikaCrawlStateYDao extends AbstractYDao {
    private static final String TABLE_NAME = "metrika_crawl_state";

    public MetrikaCrawlStateYDao() {
        super(PREFIX_METRIKA, TABLE_NAME);
    }

    public void saveState(MetrikaCrawlStateData data) {
        upsert(
                F.DOMAIN.value(data.getDomain()),
                F.COUNTER_ID.value(data.getCounterId()),
                F.STATE.value(data.getState()),
                F.LAST_UPDATE.value(data.getUpdateDate())
        ).execute();
    }

    public List<MetrikaCrawlStateData> getStatesForDomain(String domain) {
        return select(MAPPER)
                .where(F.DOMAIN.eq(domain))
                .queryForList()
                .stream().sorted(Comparator.comparingLong(MetrikaCrawlStateData::getCounterId))
                .collect(Collectors.toList());
    }

    public List<MetrikaCrawlStateData> getStatesForDomains(Collection<String> domains) {
        if (domains.isEmpty()) {
            return Collections.emptyList();
        }

        return select(MAPPER)
                .where(F.DOMAIN.in(domains))
                .queryForList();
    }

    @Nullable
    public MetrikaCrawlStateData getState(String domain, Long counterId) {
        return select(MAPPER)
                .where(F.DOMAIN.eq(domain))
                .and(F.COUNTER_ID.eq(counterId))
                .queryOne();
    }

    public void delete(String domain, long counterId) {
        delete()
                .where(F.DOMAIN.eq(domain))
                .and(F.COUNTER_ID.eq(counterId))
                .execute();
    }

    public void forEach(Consumer<MetrikaCrawlStateData> consumer) {
        streamReader(MAPPER, consumer);
    }

    private static DataMapper<MetrikaCrawlStateData> MAPPER = DataMapper.create(
            F.DOMAIN, F.COUNTER_ID, F.STATE, F.LAST_UPDATE,
            MetrikaCrawlStateData::new
    );

    private static class F {
        static final Field<String> DOMAIN = Fields.stringField("domain");
        static final Field<Long> COUNTER_ID = Fields.longField("counter_id");
        static final Field<MetrikaCounterCrawlStateEnum> STATE = Fields.stringEnumField("state", MetrikaCounterCrawlStateEnum.R);
        static final Field<DateTime> LAST_UPDATE = Fields.jodaDateTimeField("last_update");
    }
}
