package ru.yandex.webmaster3.storage.metrika.data;

import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * @author avhaliullin
 */
public class MetrikaDomainCrawlState {
    private final String domain;

    /**
     * Счетчики, обход которых разрешен
     */
    @NotNull
    private final Set<Long> enabledCounters;

    /**
     * Счетчики, обход которых приостановлен Роботом
     */
    @NotNull
    private final Set<Long> suspendedCounters;

    /**
     * Счетчики, обход которых явно был выключен пользователем
     */
    @NotNull
    private final Set<Long> disabledCounters;

    public MetrikaDomainCrawlState(
            String domain,
            @NotNull Set<Long> enabledCounters,
            @NotNull Set<Long> suspendedCounters,
            @NotNull Set<Long> disabledCounters) {
        this.domain = domain;
        this.enabledCounters = new HashSet<>(enabledCounters);
        this.suspendedCounters = new HashSet<>(suspendedCounters);
        this.disabledCounters = new HashSet<>(disabledCounters);
    }

    public MetrikaDomainCrawlState(String domain) {
        this(domain, Collections.emptySet(), Collections.emptySet(), Collections.emptySet());
    }


    public MetrikaCounterCrawlStateEnum getMetrikaCounterCrawlState(long counterId) {
        // если счетчика нет в enabled и suspended, то считаем что он выключен, disabled игнорируем намеренно
        if (enabledCounters.contains(counterId)) {
            return MetrikaCounterCrawlStateEnum.ENABLED;
        } else if (suspendedCounters.contains(counterId)) {
            return MetrikaCounterCrawlStateEnum.SUSPENDED;
        } else {
            return MetrikaCounterCrawlStateEnum.DISABLED;
        }
    }

    public void add(long counterId, MetrikaCounterCrawlStateEnum state) {
        switch (state) {
            case ENABLED:
                enabledCounters.add(counterId);
                break;
            case DISABLED:
                disabledCounters.add(counterId);
                break;
            case SUSPENDED:
                suspendedCounters.add(counterId);
                break;
        }
    }

    public String getDomain() {
        return domain;
    }

    public boolean hasEnabled() {
        return !enabledCounters.isEmpty();
    }

    @NotNull
    public Set<Long> getEnabledCounters() {
        return enabledCounters;
    }

    @NotNull
    public Set<Long> getSuspendedCounters() {
        return suspendedCounters;
    }

    @NotNull
    public Set<Long> getDisabledCounters() {
        return disabledCounters;
    }

    public Set<Long> getAllCounters() {
        Set<Long> res = new HashSet<>(enabledCounters);
        res.addAll(suspendedCounters);
        res.addAll(disabledCounters);

        return res;
    }

    public Map<Long, MetrikaCounterCrawlStateEnum> getAsMap() {
        Map<Long, MetrikaCounterCrawlStateEnum> res = new HashMap<>();
        enabledCounters.forEach(cid -> res.put(cid, MetrikaCounterCrawlStateEnum.ENABLED));
        disabledCounters.forEach(cid -> res.put(cid, MetrikaCounterCrawlStateEnum.DISABLED));
        suspendedCounters.forEach(cid -> res.put(cid, MetrikaCounterCrawlStateEnum.SUSPENDED));

        return res;
    }
}

