package ru.yandex.webmaster3.storage.mirrors.dao;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.experimental.NonFinal;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.CityHash102;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.clickhouse.system.data.ClickhouseSystemTableInfo;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.clickhouse2.*;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

@Slf4j
public abstract class AbstractNotificationsChangesCHDao extends AbstractClickhouseDao
        implements UserTakeoutDataProvider  {

    protected static final int SHARD = 0;

    @Autowired
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;

    public int countRecords(String tableId) {
        String tableName = getTable().replicatedMergeTreeTableName(0, tableId);
        Statement st = QueryBuilder.select(QueryBuilder.count().toString()).from(getTable().getDatabase(), tableName);

        return getClickhouseServer().queryOne(
                ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(SHARD)),
                st, chRow -> chRow.getLongUnsafe(0)).orElseThrow(IllegalStateException::new).intValue();
    }

    public List<AbstractMessage> getMessages(String tableId, int offset, int pageSize) throws ClickhouseException {
        ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults()
                .setHost(getClickhouseServer().pickAliveHostOrFail(SHARD));
        String tableName = getTable().replicatedMergeTreeTableName(0, tableId);

        Statement statement = QueryBuilder
                .select(getTable().getFields().stream().map(CHField::getName).collect(Collectors.toList()))
                .from(getTable().getDatabase(), tableName)
                .orderBy(getOrderBy())
                .limit(offset, pageSize);
        return getClickhouseServer().queryAll(context, statement, this::getMapper);
    }

    public void dropTable(String tableId) {
        // dropping old tables
        String query = "DROP TABLE IF EXISTS " + getTable().getDatabase() + "." + getTable().replicatedMergeTreeTableName(-1, tableId);
        getClickhouseServer().executeOnAllHosts(ClickhouseQueryContext.useDefaults(), query, SHARD);
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        List<ClickhouseHost> shardHosts = getClickhouseServer().getHosts().stream().
                filter(ClickhouseHost::isUp)
                .filter(host -> host.getShard() == SHARD)
                .toList();

        String tableNamePrefix = StringUtils.removeEnd(getTable().getName(), "%s");
        Set<String> tableNames = shardHosts.stream()
                .flatMap(host -> clickhouseSystemTablesCHDao.getTablesForPrefix(host, getTable().getDatabase(), tableNamePrefix).stream()
                        .map(ClickhouseSystemTableInfo::getName))
                .filter(tableName -> !tableName.endsWith("_distrib"))
                .collect(Collectors.toSet());

        for (var tableName : tableNames) {
            String query = String.format("ALTER TABLE %s DELETE WHERE user_id=%s",
                    getTable().getDatabase() + "." + tableName, user.getUserId());
            getClickhouseServer().execute(pickHost(tableName), query);
        }
    }

    private ClickhouseHost pickHost(String tableName) {
        Where countTablesQuery = QueryBuilder.select()
                .countAll()
                .from("system.tables")
                .where(QueryBuilder.eq("database", getTable().getDatabase()))
                .and(QueryBuilder.eq("name", tableName));

        List<ClickhouseHost> shardHosts = getClickhouseServer().getHosts().stream().
                filter(ClickhouseHost::isUp)
                .filter(host -> host.getShard() == SHARD)
                .toList();

        ClickhouseHost targetHost = null;
        for (ClickhouseHost host : shardHosts) {
            try {
                long count = getClickhouseServer().executeWithFixedHost(host, () -> queryOne(countTablesQuery,
                        r -> r.getLongUnsafe(0)).orElse(0L));
                if (count > 0) {
                    targetHost = host;
                    break;
                }
            } catch (ClickhouseException e) {
                log.error("Failed to show clickhouse tables on " + host, e);
            }
        }

        if (targetHost == null) {
            throw new WebmasterException("Get messages failed",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(getClass(), null),
                    new ClickhouseException("No clickhouse hosts avaliable for table " + tableName, countTablesQuery.toString(), null));
        }

        return targetHost;
    }

    public abstract CHTable getTable();

    public abstract List<Pair<Object, OrderBy.Direction>> getOrderBy();

    protected abstract AbstractMessage getMapper(CHRow row);

    @Value
    @AllArgsConstructor
    @NonFinal
    public static abstract class AbstractMessage {
        protected WebmasterHostId hostId;
        protected NotificationType notificationType;
        protected long userId;
        protected boolean channelEmail;
        protected boolean channelService;
        protected String email;
        protected String fio;
        protected LanguageEnum language;
        protected String login;

        public String key() {
            return String.format("%s-%s-%d", hostId.toString(), notificationType, userId);
        }

        public abstract MessageContent getContent();
    }
}
