package ru.yandex.webmaster3.storage.mirrors.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorActionEnum;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Select;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/*
 * @author kravchenko99
 * @date 16/07/21
 */
@Repository
public class MainMirrorRequestsYDao extends AbstractYDao {
    private static final String TABLE_NAME = "main_mirror_requests";

    public MainMirrorRequestsYDao() {
        super(PREFIX_MIRRORS, TABLE_NAME);
    }

    public MirrorRequest getLatestRequest(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .order(F.CREATE_DATE.desc())
                .limit(1)
                .queryOne();
    }

    public MirrorRequest getMirrorRequest(WebmasterHostId hostId, UUID requestId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .and(F.REQUEST_ID.eq(requestId))
                .limit(1)
                .queryOne();
    }

    public List<MirrorRequest> listAllRequests() {
        List<MirrorRequest> list = new ArrayList<>();
        foreach(list::add);
        return list;
    }

    public long countMirrorRequest(WebmasterHostId hostId, DateTime dateFrom, DateTime dateTo) {
        Select.Where<Long> st = countAll()
                .where(F.HOST_ID.eq(hostId));

        if (dateFrom != null && dateTo != null) {
            st = st.and(F.CREATE_DATE.gte(dateFrom))
                    .and(F.CREATE_DATE.lte(dateTo));
        }
        return st.queryOne();
    }

    public List<MirrorRequest> listMirrorRequests(WebmasterHostId hostId, DateTime dateFrom, DateTime dateTo,
                                                   MirrorRequestStateEnum state, Integer limit) {
        var st = select(MAPPER)
                .where(F.HOST_ID.eq(hostId));

        if (dateFrom != null && dateTo != null) {
            st = st.and(F.CREATE_DATE.gte(dateFrom))
                    .and(F.CREATE_DATE.lte(dateTo));
        }

        if (state != null) {
            st = st.and(F.STATE.eq(state));
        }

        if (limit != null) {
            st.limit(limit);
        }
        st.order(F.CREATE_DATE.desc());
        return st.queryForList();
    }

    public void saveRequest(MirrorRequest request) {
        upsert(VALUE_MAPPER, request).execute();
    }

    public void foreach(Consumer<MirrorRequest> consumer) {
        streamReader(MAPPER, consumer);
    }

    public void batchInsert(List<MirrorRequest> items) {
        Lists.partition(items, 2500)
                        .forEach(list ->  batchInsert(VALUE_MAPPER, list).execute());
    }

    private static final DataMapper<MirrorRequest> MAPPER = DataMapper.create(
            F.HOST_ID,
            F.REQUEST_ID,
            F.STATE,
            F.OLD_MAIN_MIRROR_HOST_ID,
            F.DESIRED_MAIN_MIRROR_HOST_ID,
            F.CREATE_DATE,
            F.UPDATE_DATE,
            F.ATTEMPT_COUNT,
            F.SERVICE_RESPONSE,
            F.ACTION,
            F.HIDDEN,
            MirrorRequest::new
    );

    private static final ValueDataMapper<MirrorRequest> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, MirrorRequest::getHostId),
            Pair.of(F.REQUEST_ID, MirrorRequest::getRequestId),
            Pair.of(F.STATE, MirrorRequest::getState),
            Pair.of(F.ACTION, MirrorRequest::getAction),
            Pair.of(F.OLD_MAIN_MIRROR_HOST_ID, MirrorRequest::getOldMainMirrorHostId),
            Pair.of(F.DESIRED_MAIN_MIRROR_HOST_ID, MirrorRequest::getNewMainMirrorHostId),
            Pair.of(F.CREATE_DATE, MirrorRequest::getCreateDate),
            Pair.of(F.UPDATE_DATE, MirrorRequest::getUpdateDate),
            Pair.of(F.ATTEMPT_COUNT, MirrorRequest::getAttemptCount),
            Pair.of(F.SERVICE_RESPONSE, MirrorRequest::getServiceResponse),
            Pair.of(F.HIDDEN, MirrorRequest::isHidden)
    );

    private static final class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<DateTime> CREATE_DATE = Fields.jodaDateTimeField("create_date");
        static final Field<UUID> REQUEST_ID = Fields.uuidField("request_id");
        static final Field<MirrorRequestStateEnum> STATE = Fields.intEnumField("state", MirrorRequestStateEnum.R);
        static final Field<Integer> ATTEMPT_COUNT = Fields.intField("attempt_count");
        static final Field<WebmasterHostId> DESIRED_MAIN_MIRROR_HOST_ID = Fields.hostIdField("desired_main_host_id");
        static final Field<WebmasterHostId> OLD_MAIN_MIRROR_HOST_ID = Fields.hostIdField("old_main_host_id");
        static final Field<DateTime> UPDATE_DATE = Fields.jodaDateTimeField("update_date");
        static final Field<String> SERVICE_RESPONSE = Fields.stringField("service_response").makeOptional();
        static final Field<MirrorActionEnum> ACTION =
                Fields.intEnumField("action", MirrorActionEnum.R).withDefault(MirrorActionEnum.MOVE);
        static final Field<Boolean> HIDDEN = Fields.boolField("hidden").withDefault(false);
    }
}
