package ru.yandex.webmaster3.storage.mobile;

import NUrlChecker.Request;
import com.datastax.driver.core.utils.UUIDs;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.logbroker.writer.LogbrokerClient;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.MobileAuditForUrlTaskData;
import ru.yandex.webmaster3.core.zora.ZoraUserAgent;
import ru.yandex.webmaster3.storage.mobile.dao.UserHostMobileAuditRequestsYDao;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditRequestInfo;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditRequestState;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditResult;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditResultType;
import ru.yandex.webmaster3.core.mobile.data.ScreenshotResolution;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckDeviceType;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckDataBlocksYDao;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.IndexingInfoData;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.PageTextContentTitleDescData;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockState;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestSource;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

/**
 * @author avhaliullin
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class MobileAuditRequestsService implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(MobileAuditRequestsService.class);

    private static final int REQUESTS_FOR_HOST_PER_DAY_LIMIT = 300;
    private static final Duration WORKER_START_TASK_TIMEOUT = Duration.standardMinutes(10);
    private static final Duration WORKER_FINISH_TASK_TIMEOUT = Duration.standardSeconds(60 * 3 + 30);
    private static final List<UrlCheckDataBlockType> blockTypes = Arrays.asList(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON,
            UrlCheckDataBlockType.ROBOT_INDEXING_INFO);

    private final UserHostMobileAuditRequestsYDao userHostMobileAuditRequestsYDao;
    private final WorkerClient workerClient;
    private final LogbrokerClient urlCheckLogbrokerClient;
    private final UrlCheckDataBlocksYDao urlCheckDataBlocksYDao;

    public long getRequestsLeftForHostToday(WebmasterHostId hostId) {
        try {
            int todayRequests = userHostMobileAuditRequestsYDao.getAllHostRequestsByCreationDesc(hostId, DateTime.now().withTimeAtStartOfDay()).size();
            return Math.max(0, REQUESTS_FOR_HOST_PER_DAY_LIMIT - todayRequests);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get requests left for host",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    private MobileAuditRequestInfo getLatestRequest(long userId, WebmasterHostId hostId) {
        try {
            DateTime now = DateTime.now();
            DateTime requestsFrom = now.minus(UserHostMobileAuditRequestsYDao.TTL);
            Optional<MobileAuditRequestInfo> requestInfoOpt =
                    userHostMobileAuditRequestsYDao.getAllHostRequestsByCreationDesc(hostId, requestsFrom)
                            .stream()
                            .filter(r -> r.getUserId() == userId && !r.isAdminRequest())
                            .findFirst();
            if (requestInfoOpt.isEmpty()) {
                log.info("No request found");
                return null;
            }

            MobileAuditRequestInfo requestInfo = requestInfoOpt.get();
            log.info("Got request: {}", requestInfo);
            switch (requestInfo.getState()) {
                case TASK_FINISHED:
                    return requestInfo;
                case NEW:
                    if (requestInfo.getLastUpdate().plus(WORKER_START_TASK_TIMEOUT).isBefore(now)) {
                        log.error("Worker failed to start request in {}", WORKER_START_TASK_TIMEOUT);
                        return requestInfo.withResult(new MobileAuditResult.InternalError());
                    } else {
                        return requestInfo;
                    }
                case TASK_STARTED:
                    if (requestInfo.getLastUpdate().plus(WORKER_FINISH_TASK_TIMEOUT).isBefore(now)) {
                        log.error("Worker failed to finish request in {}", WORKER_FINISH_TASK_TIMEOUT);
                        return requestInfo.withResult(new MobileAuditResult.InternalError());
                    } else {
                        return requestInfo;
                    }
                default:
                    throw new RuntimeException("Unknown mobile audit request state " + requestInfo.getState());
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get latest mobile audit request",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public MobileAuditRequestInfo getRequest(long userId, WebmasterHostId hostId, UUID requestId) {
        try {
            MobileAuditRequestInfo requestInfo = userHostMobileAuditRequestsYDao.getRequest(hostId, requestId);
            var dataBlocks = urlCheckDataBlocksYDao.getWithoutData(requestId);
        
            if (requestInfo == null || dataBlocks.isEmpty() || requestInfo.getUserId() != userId || 
               requestInfo.isAdminRequest()) {
                return null;
            }

            if (requestInfo.getResult() != null && requestInfo.getResult().getType() != MobileAuditResultType.OK) {
                //Если мобилопригодность провалилась, закрываем запрос в урлчекер тоже
                for (UrlCheckDataBlockType block : blockTypes) {
                    urlCheckDataBlocksYDao.closeRequest(requestId, block, UrlCheckDataBlockState.INTERNAL_ERROR);
                }
                return requestInfo;
            }

            if (requestInfo.getState() != MobileAuditRequestState.TASK_FINISHED || requestInfo.getResult() == null 
                        || !(MobileAuditResult.Success.class.isAssignableFrom(requestInfo.getResult().getClass()))) {
                return requestInfo;
            }

            if ((UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.INTERNAL_ERROR)) || 
                (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.ROBOT_FETCH_ERROR)) ||
                (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.TIMED_OUT))) {
                //Проверка в урлчекер провалилась, закрываем мобилопригодность с ошибкой тоже
                finishTask(hostId, requestId, new MobileAuditResult.InternalError());
                return userHostMobileAuditRequestsYDao.getRequest(hostId, requestId);
            }

            if (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.IN_PROGRESS)) {
                //вторая часть проверки еще в процессе
                return new MobileAuditRequestInfo(requestInfo.getUserId(), requestInfo.getHostId(), requestInfo.getUrl(), 
                requestInfo.getResolution(), requestInfo.getRequestId(), requestInfo.getLastUpdate(), MobileAuditRequestState.TASK_STARTED,
                requestInfo.getResult(), requestInfo.isAdminRequest());
            }

            //если мы пришли сюда, значит можно комбинировать и отпрвить 
            String alternateUrl, redirectTargetDesc;

            UrlCheckDataBlock robotArchiveBlock = urlCheckDataBlocksYDao.get(requestId, UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON);
            var pageData = JsonMapping.readValue(robotArchiveBlock.getData(), PageTextContentTitleDescData.class);
            UrlCheckDataBlock robotIndexingBlock = urlCheckDataBlocksYDao.get(requestId, UrlCheckDataBlockType.ROBOT_INDEXING_INFO);
            var indexingInfoData = JsonMapping.readValue(robotIndexingBlock.getData(), IndexingInfoData.class);

            alternateUrl = pageData.getLinkAlternate();
            redirectTargetDesc = indexingInfoData.getIndexingInfo().getRedirectTarget();

            MobileAuditResult mobileAuditResult = refineMobileAuditReasult(requestInfo, alternateUrl, redirectTargetDesc);
            if (mobileAuditResult == null) {
                return requestInfo;
            }
            return requestInfo.withResult(mobileAuditResult);
            

        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get latest mobile audit request",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    private MobileAuditResult refineMobileAuditReasult(MobileAuditRequestInfo requestInfo, String alternateUrl, String redirectTargetDesc) {
        MobileAuditResult.Success oldResult = ((MobileAuditResult.Success) requestInfo.getResult());
        if (oldResult == null) {
            return null;
        }
        
        URL url = requestInfo.getUrl();
        String targetUrl = oldResult.getReplaceUrl();
        String expectedUrl = getExpectedUrl(url, targetUrl);
        if (Strings.isNullOrEmpty(targetUrl)) {
            targetUrl = url.toExternalForm();
        }

        boolean hasLinkAlternate;
        boolean hasTouchDesktopLink;

        MobileAuditResult.Success.Indicators oldIndicators = oldResult.getIndicators();
        if (Strings.isNullOrEmpty(alternateUrl)) {
            if ((targetUrl.equals(url.toString()) || targetUrl.equals(redirectTargetDesc)) 
                    && isMobileFriendlyIndicators(oldIndicators)) {
                //нет ни альтернейта ни редиректа и мобилопригоден, значит адаптивная верстка
                //либо десктопный редирект совпадает с мобильным таргетом и мобилопригоден
                hasLinkAlternate = true;
                hasTouchDesktopLink = true;
            } else {
                //если альтернейта нет, связь зависит от реплейса
                hasLinkAlternate = false;
                hasTouchDesktopLink = isGoodHostReplaced(url, targetUrl);
            }
            
        } else {
            hasLinkAlternate = true;
            //если альтернейт есть, то связь зависит от его состояния
            hasTouchDesktopLink = compareUrls(alternateUrl, targetUrl);
        }

        MobileAuditResult.Success.Indicators refinedIndicators = new MobileAuditResult.Success.Indicators(oldIndicators.isHasViewPort(),
                    oldIndicators.isHasNoHorizontalScrolling(),
                    oldIndicators.isHasNoFlash(), oldIndicators.isHasNoApplet(),
                    oldIndicators.isHasNoSilverlight(), oldIndicators.isHasNoMuchSmallText(),
                    hasTouchDesktopLink, hasLinkAlternate);

        MobileAuditResult mobileAuditResult = new MobileAuditResult.Success(
                checkMobileFriendlyStatus(refinedIndicators, oldResult.isMobileFriendly(), url),
                oldResult.getBase64PNGScreenshot(),
                refinedIndicators,
                alternateUrl,
                targetUrl,
                expectedUrl
        );

        log.info("url: {};     targetUrl: {};      alternate {};      expectedUrl {};   hasTouchDesktopLink {};     desctopRedirectTarget {}", 
                    url.toExternalForm(), targetUrl, alternateUrl, expectedUrl, hasTouchDesktopLink, redirectTargetDesc);
        return mobileAuditResult;
    }

    private boolean isMobileFriendlyIndicators(MobileAuditResult.Success.Indicators indicators) {
        return indicators.isHasViewPort() && indicators.isHasNoHorizontalScrolling() &&
            indicators.isHasNoFlash() && indicators.isHasNoApplet() && indicators.isHasNoSilverlight() &&
            indicators.isHasNoMuchSmallText();
    }

    private boolean checkMobileFriendlyStatus(MobileAuditResult.Success.Indicators indicators, boolean externalMobileFriendly, URL url) {
        boolean isMobileFriendly = isMobileFriendlyIndicators(indicators);

        if (isMobileFriendly != externalMobileFriendly) {
            log.info("MobileFriendly external status mismatch indicators. False {} for {}", 
                    (externalMobileFriendly ? "positive" : "negative"), url);
        }

        return isMobileFriendly && indicators.isHasTouchDesktopLink(); //hasLinkAlternate не влияет
    }

    private boolean isGoodHostReplaced(URL url, String targetUrl) {
        if (targetUrl.equals(url.toString())) {
            return true; //no host replaced is fine
        }

        try {
            String origPath = url.getFile();
            URL target = new URL(targetUrl);
            String targetPath = target.getFile();
            return compareUrls(origPath, targetPath);

        } catch (MalformedURLException e) {
            return false;
        }
    }

    private boolean compareUrls(String srcUrl, String dstUrl) {
        //Логика сравнения из робота
        if (srcUrl.length() == dstUrl.length()) {
            return srcUrl.equals(dstUrl);
        }
        if (srcUrl.length() == dstUrl.length() + 1) {
            return srcUrl.equals(dstUrl + '/');
        }
        if (srcUrl.length() + 1 == dstUrl.length()) {
            return dstUrl.equals(srcUrl + '/');
        }

        return false;
    }

    private String getExpectedUrl(URL originalUrl, String targetUrl) {
        try {
            if (Strings.isNullOrEmpty(targetUrl)) {
                String hostName = addMPrefix(originalUrl.getHost());
                return (new URL(originalUrl.getProtocol(), hostName, originalUrl.getPort(), originalUrl.getFile())).toExternalForm();
            }
        } catch (MalformedURLException e) {}
            return targetUrl;
        
    }

    private static final String WWW_PREFIX = "www.";
    private static final String M_PREFIX = "m.";
    private String addMPrefix(String hostName) {
        if (hostName.toLowerCase().startsWith(WWW_PREFIX + M_PREFIX) || hostName.toLowerCase().startsWith(M_PREFIX)) {
            return hostName;
        }
        if (hostName.toLowerCase().startsWith(WWW_PREFIX)) {
            return WWW_PREFIX + M_PREFIX + hostName.substring(WWW_PREFIX.length());
        }
        return M_PREFIX + hostName;
    }

    public CreateRequestResult createRequest(long userId, WebmasterHostId hostId, URL url, ScreenshotResolution resolution) {
        try {
            DateTime now = DateTime.now();
            DateTime dayStart = now.withTimeAtStartOfDay();
            DateTime oldestPossiblePendingRequest = makeOldestPossiblePendingRequestDate();
            List<MobileAuditRequestInfo> requests = userHostMobileAuditRequestsYDao.getAllHostRequestsByCreationDesc(hostId, TimeUtils.earliestOf(dayStart, oldestPossiblePendingRequest));
            long requestsMadeToday = requests
                    .stream()
                    .filter(r -> !r.getCreatedAt().isBefore(dayStart) && !r.isAdminRequest())
                    .count();
            if (requestsMadeToday >= REQUESTS_FOR_HOST_PER_DAY_LIMIT) {
                return new CreateRequestResult(CreateRequestResultType.HOST_LIMIT_REACHED, null);
            }

            return new CreateRequestResult(CreateRequestResultType.OK, createRequestNoChecks(userId, hostId, url, resolution));
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to create mobile audit request",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public UUID createRequestNoChecks(long userId, WebmasterHostId hostId, URL url, ScreenshotResolution resolution) {
        try {
            DateTime now = DateTime.now();

            UUID requestId = UUIDs.timeBased();
            userHostMobileAuditRequestsYDao.createRequest(userId, hostId, requestId, now, url, resolution, false);

            if (!workerClient.checkedEnqueueTask(new MobileAuditForUrlTaskData(requestId, userId, hostId))) {
                userHostMobileAuditRequestsYDao.deleteRequest(hostId, requestId);
                
                throw new WebmasterException("Failed to create mobile audit request",
                        new WebmasterErrorResponse.WorkerErrorResponse(MobileAuditRequestsService.class, null));
            }
            
            // добавим в базу записи про все блоки данных, которые нам нужны для этого запроса
            urlCheckDataBlocksYDao.add(requestId, blockTypes, UrlCheckRequestSource.URL_MOBILE_AUDIT, DateTime.now());

            UrlCheckDeviceType deviceType = UrlCheckDeviceType.DESKTOP;
            String strUrl = url.toExternalForm();
            // отправим запрос в Роботный сервис
            var robotReqB = Request.TUrlCheckRequest.newBuilder()
                    .setTargetDevice(deviceType.getRobotDeviceType())
                    .setUrl(strUrl)
                    .setDataOrder(getRobotBlocks(blockTypes))
                    .setId(requestId.toString())
                    .setViewportWidth(resolution.getWidth())
                    .setViewportHeight(resolution.getHeight())
                    .setUserAgent(ZoraUserAgent.ROBOT.getValue())
                    .setTimestamp((int)(System.currentTimeMillis() / 1000));

            try {
                urlCheckLogbrokerClient.write(robotReqB.build().toByteArray());
            } catch (Exception e) {
                throw new WebmasterException("Failed to write to LB",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Failed to write to LB"), e);
            }

            return requestId;
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to create mobile audit request",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    @NotNull
    private Integer getRobotBlocks(Collection<UrlCheckDataBlockType> blockTypes) {
        return blockTypes.stream()
                .map(UrlCheckDataBlockType::getRobotItem)
                .filter(Objects::nonNull)
                .map(Request.EDataOrderItem::getNumber)
                .reduce(0, (v1, v2) -> v1 | v2);
    }

    public MobileAuditRequestInfo workerStartedTask(long userId, WebmasterHostId hostId, UUID requestId) {
        try {
            MobileAuditRequestInfo latestRequest = getLatestRequest(userId, hostId);
            if (latestRequest == null || !latestRequest.getRequestId().equals(requestId) || latestRequest.getState() != MobileAuditRequestState.NEW) {
                return null;
            } else {
                userHostMobileAuditRequestsYDao.updateRequest(hostId, requestId, MobileAuditRequestState.TASK_STARTED);
                return latestRequest;
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to mark task as started",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public void finishTask(WebmasterHostId hostId, UUID requestId, MobileAuditResult result) {
        try {
            userHostMobileAuditRequestsYDao.finishRequest(hostId, requestId, result.getType(), result);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to finish mobile audit task",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        userHostMobileAuditRequestsYDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                userHostMobileAuditRequestsYDao.getTablePath()
        );
    }

    private DateTime makeOldestPossiblePendingRequestDate() {
        return DateTime.now().minus(WORKER_START_TASK_TIMEOUT.plus(WORKER_FINISH_TASK_TIMEOUT));
    }

    @ToString
    public static class CreateRequestResult {
        private final CreateRequestResultType type;
        private final UUID requestId;

        public CreateRequestResult(MobileAuditRequestsService.CreateRequestResultType type, UUID requestId) {
            this.type = type;
            this.requestId = requestId;
        }

        public MobileAuditRequestsService.CreateRequestResultType getType() {
            return type;
        }

        public UUID getRequestId() {
            return requestId;
        }
    }

    public enum CreateRequestResultType {
        OK,
        HOST_LIMIT_REACHED,
//        HAVE_PENDING_REQUEST,
    }
}
