package ru.yandex.webmaster3.storage.nca;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.nca.data.CertificateIdByDomainEntry;
import ru.yandex.webmaster3.storage.nca.data.LogEntryType;
import ru.yandex.webmaster3.storage.nca.data.ParsedCertificateState;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;

/**
 * @author kravchenko99
 * @date 5/4/22
 */

@Slf4j
@Repository
@RequiredArgsConstructor
public class CtlogCertificatesCHDao extends AbstractClickhouseDao {

    private final CertificatesByIdCHDao certificatesByIdCHDao;
    private final CertificateIdByDomainCHDao certificateIdByDomainCHDao;
    private static final Function<CHRow, ParsedCertificateState> MAPPER = r -> {
        LogEntryType type = LogEntryType.R.fromValueOrNull(r.getInt(F.TYPE));
        byte[] cert = r.getBytes(F.CERTIFICATE);
        return DeserializerCertificateTransparencyLogs.parseCertificate(type, cert);
    };

    public void insert(List<CertificateIdByDomainEntry> entries) {
        certificatesByIdCHDao.insert(entries);
        certificateIdByDomainCHDao.insert(entries);
    }

    public PageWithCertificatesAndCount getCertificates(String domain) {
        String reversedDomain = NcaRulesService.reverseDomain(domain);
        List<String> split = Lists.reverse(List.of(domain.split("\\.")));
        List<String> domainsForCheck = getDomainsForCheck(reversedDomain, split);
        ClickhouseQueryContext.Builder chContext = getCHContext(reversedDomain);

        var certificateStates = getParsedCertificateStates(reversedDomain, domainsForCheck,
                null, chContext);
        return new PageWithCertificatesAndCount(certificateStates.size(), certificateStates);
    }

    public PageWithCertificatesAndCount getCertificates(String domain, PageUtils.Pager pager) {
        String reversedDomain = NcaRulesService.reverseDomain(domain);
        List<String> split = Lists.reverse(List.of(domain.split("\\.")));
        List<String> domainsForCheck = getDomainsForCheck(reversedDomain, split);

        ClickhouseQueryContext.Builder chContext = getCHContext(reversedDomain);

        Statement countStatement = certificateIdByDomainCHDao.getCertificateIdsByDomainCount(reversedDomain,
                domainsForCheck);


        long count = getClickhouseServer().queryOne(
                chContext,
                countStatement, chRow -> chRow.getLongUnsafe(0)
        ).orElseThrow(IllegalStateException::new);

        if (count == 0) {
            return new PageWithCertificatesAndCount(0, new ArrayList<>());
        }

        List<ParsedCertificateState> certificateStates = getParsedCertificateStates(reversedDomain,
                domainsForCheck, pager, chContext);
        return new PageWithCertificatesAndCount(count, certificateStates);

    }

    @NotNull
    private List<ParsedCertificateState> getParsedCertificateStates(String reversedDomain,
                                                                    List<String> domainsForCheck,
                                                                    PageUtils.Pager pager,
                                                                    ClickhouseQueryContext.Builder chContext) {
        Statement certificateIds = certificateIdByDomainCHDao.getCertificateIdsByDomain(reversedDomain, domainsForCheck, pager);
        Statement query = certificatesByIdCHDao.getCertificatesById(certificateIds);

        List<ParsedCertificateState> parsedCertificateStates = getClickhouseServer().queryAll(
                chContext,
                query,
                MAPPER
        );

        return parsedCertificateStates.stream()
                .sorted(Comparator.comparing(ParsedCertificateState::getNotBefore).reversed())
                .collect(Collectors.toList());
    }

    @NotNull
    private static List<String> getDomainsForCheck(String reversedDomain, List<String> split) {
        List<String> domainsForCheck = new ArrayList<>();
        //чтобы вот такое не находилось ster.mail.paymentsystems.ru при запросе payments.ru
        domainsForCheck.add(reversedDomain);
        StringBuilder s = new StringBuilder(split.get(0));
        for (int i = 1; i < split.size() - 1 && i <= 4; i++) {
            s.append(".").append(split.get(i));
            domainsForCheck.add(s + ".*");
        }
        return domainsForCheck;
    }

    private ClickhouseQueryContext.Builder getCHContext(String reverseDomain) {
        int shard = certificateIdByDomainCHDao.getShardByReversedSecondLevelDomain(reverseDomain);
        return ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(shard));
    }

    public record PageWithCertificatesAndCount(long count, List<ParsedCertificateState> certificateStates){};

    // копия в AbstractCtlogCertificatesCHDao
    private interface F {
        //
        String CERTIFICATE = "certificate";
        String TYPE = "type";

    }
}
