package ru.yandex.webmaster3.storage.notifications;

import com.google.protobuf.ByteString;
import ru.yandex.webmaster3.proto.notifications.Notifications;

import java.util.BitSet;

/**
 * @author avhaliullin
 */
public class NotificationProgress {
    private boolean sending;
    private final int chunkSize;
    private final int usersCount;
    private final BitSet finishedChunks;
    private int revisionId;
    private long lastStarted;
    private int startWithFinishedChunks;
    private long firstChunkAfterLastStartTs;
    private long lastChunkTs;

    public NotificationProgress(boolean sending, int chunkSize, int usersCount, BitSet finishedChunks, int revisionId,
                                long lastStarted, int startWithFinishedChunks, long firstChunkAfterLastStartTs,
                                long lastChunkTs) {
        this.sending = sending;
        this.chunkSize = chunkSize;
        this.usersCount = usersCount;
        this.finishedChunks = finishedChunks;
        this.revisionId = revisionId;
        this.lastStarted = lastStarted;
        this.startWithFinishedChunks = startWithFinishedChunks;
        this.firstChunkAfterLastStartTs = firstChunkAfterLastStartTs;
        this.lastChunkTs = lastChunkTs;
    }

    public long getFirstChunkAfterLastStartTs() {
        return firstChunkAfterLastStartTs;
    }

    public void setFirstChunkAfterLastStartTs(long firstChunkAfterLastStartTs) {
        this.firstChunkAfterLastStartTs = firstChunkAfterLastStartTs;
    }

    public long getLastChunkTs() {
        return lastChunkTs;
    }

    public void setLastChunkTs(long lastChunkTs) {
        this.lastChunkTs = lastChunkTs;
    }

    public void setLastStarted(long lastStarted) {
        this.lastStarted = lastStarted;
    }

    public long getLastStarted() {
        return lastStarted;
    }

    public int getStartWithFinishedChunks() {
        return startWithFinishedChunks;
    }

    public void setStartWithFinishedChunks(int startWithFinishedChunks) {
        this.startWithFinishedChunks = startWithFinishedChunks;
    }

    public void setRevisionId(int revisionId) {
        this.revisionId = revisionId;
    }

    public int getRevisionId() {
        return revisionId;
    }

    public int getChunkSize() {
        return chunkSize;
    }

    public int getUsersCount() {
        return usersCount;
    }

    public BitSet getFinishedChunks() {
        return finishedChunks;
    }

    public int getChunksCount() {
        return usersCount == 0 ? 0 : (usersCount + chunkSize - 1) / chunkSize;
    }

    public boolean isSending() {
        return sending;
    }

    public void setSending(boolean sending) {
        this.sending = sending;
    }

    private static BitSet makeBitSet(int usersCount, int chunkSize) {
        if (usersCount == 0) {
            return new BitSet(0);
        }
        int chunksCount = (usersCount + chunkSize - 1) / chunkSize;
        return new BitSet(chunksCount);
    }

    public Notifications.NotificationProgress convertToProto() {
        Notifications.NotificationProgress.Builder builder = Notifications.NotificationProgress.newBuilder();
        builder.setChunkSize(chunkSize)
                .setUsersCount(usersCount)
                .setFinishedChunks(ByteString.copyFrom(finishedChunks.toByteArray()))
                .setSending(sending)
                .setRevisionId(revisionId)
                .setLastStarted(lastStarted)
                .setStartWithFinishedChunks(startWithFinishedChunks)
                .setFirstChunkAfterLastStartTs(firstChunkAfterLastStartTs)
                .setLastChunkTs(lastChunkTs);
        return builder.build();
    }

    public static NotificationProgress convertFromProto(Notifications.NotificationProgress message) {
        return new NotificationProgress(
                message.getSending(),
                message.getChunkSize(),
                message.getUsersCount(),
                BitSet.valueOf(message.getFinishedChunks().toByteArray()),
                message.getRevisionId(),
                message.getLastStarted(),
                message.getStartWithFinishedChunks(),
                message.hasFirstChunkAfterLastStartTs() ? message.getFirstChunkAfterLastStartTs() : 0L,
                message.hasLastChunkTs() ? message.getLastChunkTs() : 0L);
    }

    public static NotificationProgress createInitial(int chunkSize, int usersCount, int revisionId) {
        return new NotificationProgress(true, chunkSize, usersCount, makeBitSet(usersCount, chunkSize), revisionId, System.currentTimeMillis(), 0, 0L, 0L);
    }
}
