package ru.yandex.webmaster3.storage.notifications;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

/**
 * @author aherman
 */
public class UserHostNotificationConfiguration {
    private final String email;
    private final Set<Pair<NotificationType, NotificationChannel>> enabledByDefault;
    private final Set<WebmasterHostId> hostsWithSettings;

    private final Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> enabledExplicitly;
    private final Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> disabledExplicitly;

    public UserHostNotificationConfiguration(String email,
            Set<Pair<NotificationType, NotificationChannel>> enabledByDefault,
            Set<WebmasterHostId> hostsWithSettings,
            Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> enabledExplicitly,
            Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> disabledExplicitly)
    {
        this.email = email;
        this.enabledByDefault = enabledByDefault;
        this.hostsWithSettings = hostsWithSettings;
        this.enabledExplicitly = enabledExplicitly;
        this.disabledExplicitly = disabledExplicitly;
    }

    public String getEmail() {
        return email;
    }

    public HostNotificationMode getPerUserSetting(NotificationType type, NotificationChannel channel) {
        if (enabledByDefault.contains(Pair.of(type, channel))) {
            return HostNotificationMode.ENABLED;
        }
        return HostNotificationMode.DISABLED;
    }

    public Optional<HostNotificationMode> getPerHostSettingOrDefault(WebmasterHostId hostId, NotificationType type,
            NotificationChannel channel)
    {
        if (!hostsWithSettings.contains(hostId)) {
            return Optional.empty();
        }
        Pair<NotificationType, NotificationChannel> typeChannel = Pair.of(type, channel);
        boolean enabled = enabledExplicitly.getOrDefault(hostId, Collections.emptySet()).contains(typeChannel);
        if (enabled) {
            return Optional.of(HostNotificationMode.ENABLED);
        }
        boolean disabled = disabledExplicitly.getOrDefault(hostId, Collections.emptySet()).contains(typeChannel);
        if (disabled) {
            return Optional.of(HostNotificationMode.DISABLED);
        }
        return Optional.of(HostNotificationMode.DEFAULT);
    }

    public Optional<HostNotificationMode> getPerHostSettingResolved(WebmasterHostId hostId, NotificationType type,
            NotificationChannel channel)
    {
        Optional<HostNotificationMode> modeOrDefault = getPerHostSettingOrDefault(hostId, type, channel);
        if (!modeOrDefault.isPresent()) {
            return modeOrDefault;
        }
        if (modeOrDefault.get() != HostNotificationMode.DEFAULT) {
            return modeOrDefault;
        }
        return Optional.of(getPerUserSetting(type, channel));
    }
}
