package ru.yandex.webmaster3.storage.notifications.dao;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

/**
 * @author avhaliullin
 */
public class NotificationCHTablesManager extends AbstractNotificationDao {
    private static final Logger log = LoggerFactory.getLogger(NotificationCHTablesManager.class);

    private static final String UUID_REGEX = "[0-9a-f]{32}";
    private static final Map<NotificationTableType, Pattern> PATTERNS;

    private static Pattern makePattern(String tableNamePrefix) {
        return Pattern.compile("^" + tableNamePrefix + "(" + UUID_REGEX + ")_(" + UUID_REGEX + ")$");
    }

    static {
        PATTERNS = new EnumMap<>(NotificationTableType.class);
        PATTERNS.put(NotificationTableType.SEARCH_BASE, makePattern(SearchBaseNotificationListCHDao.TABLE_NAME_PREFIX));
    }

    private static UUID parseUUID(String hex) {
        return UUID.fromString(hex.substring(0, 8) + "-" + hex.substring(8, 12) + "-" + hex.substring(12, 16) + "-" + hex.substring(16, 20) + "-" + hex.substring(20));
    }

    public List<NotificationTableInfo> listNotificationTables() throws ClickhouseException {
        Set<String> tableNames = new HashSet<>();
        String q = "SHOW TABLES FROM " + DB_NAME;
        for (ClickhouseHost clickhouseHost : getClickhouseServer().getHosts()) {
            ClickhouseQueryContext.Builder chContext = ClickhouseQueryContext.useDefaults().setHost(clickhouseHost);
            getClickhouseServer().execute(
                    chContext,
                    ClickhouseServer.QueryType.SELECT,
                    q,
                    Optional.empty(),
                    Optional.of(row -> {
                        tableNames.add(row.getString("name"));
                        return null;
                    })
            );
        }
        List<NotificationTableInfo> result = new ArrayList<>();
        for (String name : tableNames) {
            boolean added = false;
            for (Map.Entry<NotificationTableType, Pattern> entry : PATTERNS.entrySet()) {
                Matcher m = entry.getValue().matcher(name);
                if (m.find()) {
                    result.add(new NotificationTableInfo(entry.getKey(), parseUUID(m.group(1)), parseUUID(m.group(2))));
                    added = true;
                    break;
                }
            }
            if (!added) {
                log.error("Unknown table in notifications database: " + DB_NAME + "." + name);
            }
        }
        return result;
    }

    public void deleteTables(List<NotificationTableInfo> tables) throws ClickhouseException {
        for (NotificationTableInfo tableInfo : tables) {
            String tableName = switch (tableInfo.getTableType()) {
                case SEARCH_BASE -> SearchBaseNotificationListCHDao.TABLE_NAME_PREFIX;
                default -> throw new RuntimeException("Unknown table type " + tableInfo.getTableType());
            };
            tableName = tableName + inTableName(tableInfo.getNotificationId()) + "_" + inTableName(tableInfo.getListId());
            log.info("Dropping table " + DB_NAME + "." + tableName);
            String q = "DROP TABLE IF EXISTS " + DB_NAME + "." + tableName;
            for (ClickhouseHost host : getClickhouseServer().getHosts()) {
                ClickhouseQueryContext.Builder chContext = ClickhouseQueryContext.useDefaults().setHost(host);
                try {
                    getClickhouseServer().execute(chContext, ClickhouseServer.QueryType.INSERT, q, Optional.empty(), Optional.empty());
                } catch (ClickhouseException e) {
                    log.error("Failed to delete table " + tableName + " from clickhouse host " + host.getHostURI(), e);
                }
            }
        }
    }

}
