package ru.yandex.webmaster3.storage.notifications.service;

import java.io.UnsupportedEncodingException;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.activation.DataSource;
import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MailDateFormat;
import javax.mail.internet.MimeMessage;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.MimeMailMessage;
import org.springframework.mail.javamail.MimeMessageHelper;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.wmtools.common.service.MailService;

/**
 * @author avhaliullin
 */
public class EmailSenderService {
    private static final Logger log = LoggerFactory.getLogger(EmailSenderService.class);

    private static final String ENCODING = "UTF-8";
    private static final String EMAIL_PREFIX = "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">" +
            "<html><head><meta content=\"text/html; charset=UTF-8\" http-equiv=\"Content-Type\"></head>" +
            "<body text=\"#000000\" bgcolor=\"#ffffff\">";
    private static final String EMAIL_SUFFIX = "</body></html>";
    private static final String YAWEBMASTER = "yawebmaster";

    private MailService mailService;

    private boolean productionMode = false;
    private Set<String> devModeWhiteList = Collections.emptySet();
    private String fromEmail;
    private String fromName;


    public boolean sendEmail(String address, String toName, String subject, String content) {
        return sendEmail(address, toName, subject, content, Collections.emptyMap());
    }

    /**
     * Возвращает false, если отправка не удалась, но можно попробовать повторить.
     * Если вылетел exception - нельзя понять, ушло ли сообщение - повторять небезопасно.
     */
    public boolean sendEmail(String address, String toName, String subject, String content, Map<String, DataSource> inlines) {
        address = StringUtils.trimToEmpty(address).toLowerCase();

        // TEMP
        if (address.equals("-lennon@yandex.ru")) {
            return true;
        }

        // Иногда мы получаем кривой email адрес
        if (!isEmailValid(address)) {
            log.error("Invalid email: " + address);
            // больше пытаться не будем
            return true;
        }

        if (!productionMode) {
            if (!devModeWhiteList.contains(address) && !address.endsWith("@yandex-team.ru")) {
                log.info("Pretending send email to {} {}, Subject: {}, Body: {}", address, toName, subject, content);
                return true;
            }
            // костыль для тестинга, подменяем webmaster.yandex.ru на webmaster.test.yandex.ru
            content = content.replaceAll("webmaster.yandex.ru", "webmaster.test.yandex.ru");
        }

        log.info("Sending email to " + address);
        MimeMailMessage mimeMailMessage;
        try {
            MimeMessage message = mailService.getMailSender().createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true, ENCODING);
            InternetAddress from = new InternetAddress(fromEmail, fromName, ENCODING);

            helper.setTo(new InternetAddress(address, toName));

            helper.setFrom(new InternetAddress(fromEmail, fromName, ENCODING));

            // в сабжекте не должно быть переводов строки
            subject = subject.replace('\n', ' ');
            if (!productionMode) {
                subject = subject + " (TESTING)";
            }
            helper.setSubject(subject);
            final String html = EMAIL_PREFIX + content + EMAIL_SUFFIX;
            helper.setText(html, true);
            if (inlines != null) {
                for (Map.Entry<String, DataSource> inline : inlines.entrySet()) {
                    helper.addInline(inline.getKey(), inline.getValue());
                }
            }

            mimeMailMessage = new MimeMailMessage(message);

            final Date date = new Date();
            message.setSentDate(date);
            final MailDateFormat mdf = new MailDateFormat();
            final String sentDate = mdf.format(date);
            final String xYandexService = mailService.getYandexServiceHeaderValue(sentDate, from.toUnicodeString(), subject, YAWEBMASTER);
            message.addHeader("X-Yandex-Service", xYandexService);

        } catch (UnsupportedEncodingException | MessagingException e) {
            log.error("Failed to send message to " + address, e);
            return false;
        }
        try {
            mailService.getMailSender().send(mimeMailMessage.getMimeMessage());
            return true;
        } catch (MailException e) {
            log.error("Failed to send email to " + address, e);
            throw new WebmasterException("Failed to send email to " + address,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
        }
    }

    private boolean isEmailValid(String email) {
        boolean isValid = false;
        try {
            new InternetAddress(email).validate();
            isValid = true;
        } catch (AddressException e) {
            // ignore
        }

        return isValid;
    }


    @Required
    public void setMailService(MailService mailService) {
        this.mailService = mailService;
    }

    @Required
    public void setProductionMode(boolean productionMode) {
        this.productionMode = productionMode;
    }

    public void setDevModeWhiteList(String whiteList) {
        devModeWhiteList = new HashSet<>();
        if (!StringUtils.isEmpty(whiteList)) {
            String[] emails = whiteList.split(",");
            for (String email : emails) {
                devModeWhiteList.add(StringUtils.trimToEmpty(email).toLowerCase());
            }
        }
    }

    @Required
    public void setFromEmail(String fromEmail) {
        this.fromEmail = fromEmail;
    }

    @Required
    public void setFromName(String fromName) {
        this.fromName = fromName;
    }
}
