package ru.yandex.webmaster3.storage.notifications.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.web.util.UriComponentsBuilder;
import ru.yandex.webmaster3.storage.notifications.EmailTrackingInfo;
import ru.yandex.webmaster3.storage.notifications.EmailTrackingInfoSigner;

import javax.activation.DataSource;
import java.net.URI;
import java.util.Collections;
import java.util.Map;

/*
 * @author leonidrom
 */
public class TrackingEmailSenderService  {
    private static final Logger log = LoggerFactory.getLogger(TrackingEmailSenderService.class);

    private final static String TRACKING_TAG_FORMAT = "<img src=\"%s\" width=\"0\" height=\"0\" style=\"display: none; visibility: hidden; opacity: 0; left: -9999px; top: -9999px;\">";

    private EmailSenderService emailSenderService;
    private EmailTrackingInfoSigner emailTrackingInfoSigner;
    private URI trackerUri;

    public boolean sendEmail(String address, String toName, String subject, String content, EmailTrackingInfo trackingInfo) {
        return sendEmail(address, toName, subject, content, Collections.emptyMap(), trackingInfo);
    }

    public boolean sendEmail(String address, String toName, String subject, String content,
                             Map<String, DataSource> inlines, EmailTrackingInfo trackingInfo) {

        String trackingInfoDigest = emailTrackingInfoSigner.getDigest(trackingInfo);
        String trackingContent = getTrackingTag(trackingInfo, trackingInfoDigest, content);

        // do not delete, used in yql script!
        log.info("Email tracking: hostId={} address={} messageType={} timestamp={} digest={}",
            trackingInfo.getHostId() == null ? "EMPTY" : trackingInfo.getHostId(),
            address,
            trackingInfo.getMessageType(),
            trackingInfo.getTimestamp(),
            trackingInfoDigest
        );

        return emailSenderService.sendEmail(address, toName, subject, trackingContent, inlines);
    }

    private String getTrackingTag(EmailTrackingInfo trackingInfo, String trackingInfoDigest, String content) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromUri(trackerUri);
        trackingInfo.addToQueryParams(builder);
        builder.queryParam("digest", trackingInfoDigest);

        URI trackingLink = builder.build().toUri();
        String trackingTag = String.format(TRACKING_TAG_FORMAT, trackingLink);
        String trackingContent = trackingTag + content;

        return trackingContent;
    }

    @Required
    public void setEmailSenderService(EmailSenderService emailSenderService) {
        this.emailSenderService = emailSenderService;
    }

    @Required
    public void setEmailTrackingInfoSigner(EmailTrackingInfoSigner emailTrackingInfoSigner) {
        this.emailTrackingInfoSigner = emailTrackingInfoSigner;
    }

    @Required
    public void setTrackerUri(URI trackerUri) {
        this.trackerUri = trackerUri;
    }
}
