package ru.yandex.webmaster3.storage.notifications.service;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.UserHostNotificationConfiguration;
import ru.yandex.webmaster3.storage.notifications.UserNotificationConfiguration;
import ru.yandex.webmaster3.storage.settings.history.SettingsHistoryService;
import ru.yandex.webmaster3.storage.user.UserInitializationInfo;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.UserTakeoutTableData;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationChannelsYDao;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationEmailYDao;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationHostSettingsYDao;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationSetting;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.notification.Setting;
import ru.yandex.webmaster3.storage.user.notification.UserEmailSettings;
import ru.yandex.webmaster3.storage.user.notification.UserSetEmailInfo;
import ru.yandex.webmaster3.storage.user.service.InitializedUsersService;

/**
 * @author avhaliullin
 */
@AllArgsConstructor(onConstructor_ = @Autowired)
@Service
public class UserNotificationSettingsService implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(UserNotificationSettingsService.class);

    private static final String USER_TAKEOUT_NOTIFICATION_SETTINGS_LABEL = "notificationHostsSettings";
    private static final String USER_TAKEOUT_EMAILS_LABEL = "notificationEmailSettings";
    private static final String USER_TAKEOUT_CHANNELS_LABEL = "notificationChannelsSettings";
    private static final String USER_TAKEOUT_INITIALIZED_USER_LABEL = "initializedUser";

    private UserNotificationChannelsYDao userNotificationChannelsYDao;
    private UserNotificationEmailYDao userNotificationEmailYDao;
    private UserNotificationHostSettingsYDao userNotificationHostSettingsYDao;
    private BlackboxUsersService blackboxExternalYandexUsersService;
    private SettingsHistoryService settingsHistoryService;
    private InitializedUsersService initializedUsersService;

    public static final EnumSet<NotificationType> PER_HOST_NOTIFICATION_TYPES = EnumSet.of(
            NotificationType.SEARCH_BASE_UPDATE,
            NotificationType.SERP_LINKS_UPDATE,
            NotificationType.MAIN_MIRROR_UPDATE,
            NotificationType.SITE_PROBLEM_FATAL,
            NotificationType.SITE_PROBLEM_CRITICAL,
            NotificationType.SITE_PROBLEM_POSSIBLE,
            NotificationType.SITE_PROBLEM_RECOMMENDATION,
            NotificationType.URL_TITLE_CHANGE,
            NotificationType.URL_INDEXING_STATUS_CHANGE,
            NotificationType.URL_INDEXING_LAST_ACCESS_CHANGE,
            NotificationType.URL_SEARCH_STATUS_CHANGE,
            NotificationType.URL_SEARCH_LAST_ACCESS_CHANGE,
            NotificationType.URL_DESCRIPTION_CHANGE,
            NotificationType.URL_REL_CANONICAL_TARGET_CHANGE,
            NotificationType.DIGEST,
            NotificationType.DIGEST_LITE,
            NotificationType.SITE_ACCESS,
            NotificationType.ACCESS_LOST,
            NotificationType.METRIKA_COUNTERS,
            NotificationType.SITE_REGIONS,
            NotificationType.RECOMMENDED_QUERIES,
            NotificationType.SITE_DISPLAY_NAME,
            NotificationType.TURBO_BAN,
            NotificationType.TURBO_ERROR,
            NotificationType.TURBO_WARNING,
            NotificationType.ROBOTS_TXT_CHANGE,
            NotificationType.NEW_DOMAINS_NOTIFICATION,
            NotificationType.TURBO_ADV_SETTINGS_CHANGE,
            NotificationType.TURBO_COMMERCE_SETTINGS_CHANGE,
            NotificationType.IMPORTANT_URLS_AUTO_ADD,
            NotificationType.UNVERIFIED_HOST_REMOVAL_AFTER_WEEK,
            NotificationType.UNVERIFIED_HOST_REMOVAL_AFTER_THREE_WEEK,
            NotificationType.NEW_REVIEW_AVAILABLE,
            NotificationType.IKS_UPDATE,
            NotificationType.TURBO_SCC_BANNED,
            NotificationType.TURBO_SCC_UNBANNED,
            NotificationType.TURBO_SCC_FAILED,
            NotificationType.TURBO_SCC_PASS,
            NotificationType.VIDEOHOST_OFFER,
            NotificationType.FEEDS_INFO
    );

    public static final EnumSet<NotificationType> SERVICE_NOTIFICATION = EnumSet.of(
            NotificationType.TURBO_LISTINGS_NEW, NotificationType.TURBO_NEW
    );
    public static final EnumSet<NotificationType> EUROPEAN_SERVICE_NOTIFICATION =
            EnumSet.noneOf(NotificationType.class);

    private static final EnumSet<NotificationType> DISABLED_NOTIFICATION_TYPES = EnumSet.of(
            NotificationType.TRENDS, NotificationType.TURBO_NEW
    );

    public static final EnumSet<NotificationType> GLOBAL_NOTIFICATION_TYPES = EnumSet.complementOf(
            PER_HOST_NOTIFICATION_TYPES);

    /**
     * Не вызывать из worker'а, ходит в паспорт на каждый вызов
     *
     * @param userId
     * @return
     */
    public UserEmailSettings getEmailSettingsUI(long userId) {
        UserSetEmailInfo emailSettings = userNotificationEmailYDao.getEmailSettings(userId);
        Set<String> validatedEmails = new HashSet<>(blackboxExternalYandexUsersService.getValidatedEmails(userId));
        if (emailSettings != null && emailSettings.getEmail() != null) {
            boolean verified = validatedEmails.contains(emailSettings.getEmail());
            if (!verified) {
                verified = blackboxExternalYandexUsersService.isEmailValid(userId, emailSettings.getEmail());
                if (verified) {
                    validatedEmails.add(emailSettings.getEmail());
                }
            }
            if (verified != emailSettings.isVerified()) {
                log.info("Updating user {} email {} verified state, new state: {}", userId,
                        emailSettings.getEmail(), verified);
                userNotificationEmailYDao.updateVerifiedState(userId, emailSettings.getEmail(), verified);
                emailSettings = new UserSetEmailInfo(emailSettings.getEmail(), verified);
            }
        } else {
            emailSettings = new UserSetEmailInfo(null, false);
        }
        return new UserEmailSettings(emailSettings, validatedEmails);
    }

    @NotNull
    public Map<NotificationType, Set<NotificationChannel>> getDefaultSettings(long userId) {
        Map<NotificationType, Set<NotificationChannel>> type2channels = userNotificationChannelsYDao.getUserSettings(
                userId);
        Map<NotificationType, Set<NotificationChannel>> channelsView = new EnumMap<>(NotificationType.class);
        for (NotificationType notificationType : NotificationType.values()) {
            if (DISABLED_NOTIFICATION_TYPES.contains(notificationType)) {
                continue;
            }
            Set<NotificationChannel> channels = type2channels.getOrDefault(notificationType,
                    notificationType.getDefaultChannels());
            channelsView.put(notificationType, channels);
        }
        return channelsView;
    }

    public Set<NotificationChannel> updateDefaultSettings(WebmasterUser user, NotificationType type,
                                                          NotificationChannel channel, HostNotificationMode mode) {
        Map<NotificationType, Set<NotificationChannel>> userSettings = userNotificationChannelsYDao.getUserSettings(user.getUserId());

        Set<NotificationChannel> settings = new HashSet<>(userSettings.getOrDefault(type, type.getDefaultChannels()));
        if (mode == HostNotificationMode.DISABLED) {
            settings.remove(channel);
        } else if (mode == HostNotificationMode.ENABLED) {
            settings.add(channel);
        }
        userNotificationChannelsYDao.updateSettings(user.getUserId(), type, settings);

        settingsHistoryService.updateHostNotificationDefault(user.getUserId(),
                new Setting(type, channel, mode));
        return settings;
    }


    public UserNotificationConfiguration getUserNotificationsSettings(long userId, NotificationType notificationType) {
        UserInitializationInfo userInitializationInfo = initializedUsersService.getUserInfo(userId);
        if (userInitializationInfo == null || !userInitializationInfo.initialized()) {
            return null;
        }

        //String email = getUserEmailOrDefaultIfEmpty(userId);
        String email = userNotificationEmailYDao.getEmail(userId);
        Map<NotificationType, Set<NotificationChannel>> channelsMap = userNotificationChannelsYDao.getUserSettings(userId);
        Set<NotificationChannel> channels = channelsMap.getOrDefault(notificationType, notificationType.getDefaultChannels());
        if (email == null && !channels.isEmpty()) {
            channels = EnumSet.copyOf(channels);
            channels.remove(NotificationChannel.EMAIL);
        }
        return new UserNotificationConfiguration(channels, email);
    }

    public void saveEmail(long userId, String email) {
        userNotificationEmailYDao.saveEmail(userId, email);
        settingsHistoryService.updateNotificationEmail(userId, email);
    }

    public UserHostNotificationConfiguration getUserNotificationsSettings(WebmasterUser user, List<WebmasterHostId> hostIds) {
        UserInitializationInfo userInitializationInfo = initializedUsersService.getUserInfo(user.getUserId());
        if (userInitializationInfo == null || !userInitializationInfo.initialized()) {
            return null;
        }

        //String email = getUserEmailOrDefaultIfEmpty(user.getUserId());
        String email = userNotificationEmailYDao.getEmail(user.getUserId());

        Map<NotificationType, Set<NotificationChannel>> userDefaultSettings = userNotificationChannelsYDao.getUserSettings(user.getUserId());
        Set<Pair<NotificationType, NotificationChannel>> enabledByDefault = new HashSet<>();
        for (NotificationType type : NotificationType.values()) {
            Set<NotificationChannel> channels = userDefaultSettings.getOrDefault(type, type.getDefaultChannels());
            for (NotificationChannel channel : channels) {
                enabledByDefault.add(Pair.of(type, channel));
            }
        }

        Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> enabledExplicitly = new HashMap<>();
        Map<WebmasterHostId, Set<Pair<NotificationType, NotificationChannel>>> disabledExplicitly = new HashMap<>();
        if (!hostIds.isEmpty()) {
            List<HostNotificationSetting> hostSettings = userNotificationHostSettingsYDao.getSettings(user, hostIds);
            for (HostNotificationSetting hostSetting : hostSettings) {
                Pair<NotificationType, NotificationChannel> typeChannel =
                        Pair.of(hostSetting.getType(), hostSetting.getChannel());
                if (!PER_HOST_NOTIFICATION_TYPES.contains(hostSetting.getType())) {
                    // Ignore existing setting, if notification excluded from host notifications
                    continue;
                }

                Set<Pair<NotificationType, NotificationChannel>> settingsSet;
                switch (hostSetting.getMode()) {
                    case ENABLED:
                        settingsSet = enabledExplicitly.computeIfAbsent(hostSetting.getHostId(),
                                k -> new HashSet<>());
                        break;

                    case DISABLED:
                        settingsSet = disabledExplicitly.computeIfAbsent(hostSetting.getHostId(),
                                k -> new HashSet<>());
                        break;

                    default:
                        // Should never happen
                        throw new WebmasterException("Unsupported notification mode: " + hostSetting.getMode(),
                                new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(),
                                        "Unsupported notification mode: " + hostSetting.getMode()));
                }
                settingsSet.add(typeChannel);
            }
        }

        return new UserHostNotificationConfiguration(email, enabledByDefault, new HashSet<>(hostIds),
                enabledExplicitly, disabledExplicitly);
    }

    public String getDefaultEmail(long userId) {
        return blackboxExternalYandexUsersService.getDefaultEmail(userId);
    }

    public String getUserEmailOrDefaultIfEmpty(long userId) {
        String email = userNotificationEmailYDao.getEmail(userId);
        if (email != null) {
            return email;
        } else {
            return blackboxExternalYandexUsersService.getDefaultEmail(userId);
        }
    }

    public String getUserEmail(long userId) {
        return userNotificationEmailYDao.getEmail(userId);
    }

    public void updateHostSettings(long userId, WebmasterHostId hostId, NotificationType type,
                                   NotificationChannel channel,
                                   HostNotificationMode mode) {
        userNotificationHostSettingsYDao.updateMode(new WebmasterUser(userId), hostId, type, channel, mode);
        settingsHistoryService.updateHostNotificationSetting(userId, hostId, new Setting(type, channel, mode));
    }

    public boolean shouldAlertNoEmail(long userId, NotificationType type, NotificationChannel channel,
                                      HostNotificationMode mode) {
        if (channel == NotificationChannel.EMAIL) {
            boolean shouldCheckEmail;
            switch (mode) {
                case DISABLED:
                    shouldCheckEmail = false;
                    break;
                case ENABLED:
                    shouldCheckEmail = true;
                    break;
                case DEFAULT:
                    shouldCheckEmail = userNotificationChannelsYDao.getUserSettings(userId).getOrDefault(type,
                                    type.getDefaultChannels())
                            .contains(NotificationChannel.EMAIL);
                    break;
                default:
                    throw new RuntimeException("Unknown host notification mode " + mode);
            }
            if (shouldCheckEmail) {
                return userNotificationEmailYDao.getEmail(userId) == null;
            }
        }
        return false;
    }

    @Override
    @NotNull
    public List<UserTakeoutTableData> getUserTakeoutData(WebmasterUser user) {
        List<UserTakeoutTableData> takeoutData = new ArrayList<>();

        List<HostNotificationSetting> userSettings = userNotificationHostSettingsYDao.getSettings(user);
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_NOTIFICATION_SETTINGS_LABEL, userSettings));

        UserSetEmailInfo emailSettings = userNotificationEmailYDao.getEmailSettings(user.getUserId());
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_EMAILS_LABEL, emailSettings));

        Map<NotificationType, Set<NotificationChannel>> channelSettings = userNotificationChannelsYDao.getUserSettings(user.getUserId());
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_CHANNELS_LABEL, channelSettings));

        UserInitializationInfo initInfo = initializedUsersService.getUserInfo(user.getUserId());
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_INITIALIZED_USER_LABEL, initInfo));

        return takeoutData;
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        long userId = user.getUserId();
        userNotificationHostSettingsYDao.deleteForUser(userId);
        userNotificationEmailYDao.deleteForUser(userId);
        userNotificationChannelsYDao.deleteForUser(userId);
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                userNotificationHostSettingsYDao.getTablePath(),
                userNotificationEmailYDao.getTablePath(),
                userNotificationChannelsYDao.getTablePath()
        );
    }
}
