package ru.yandex.webmaster3.storage.originaltext;

import com.google.common.primitives.Longs;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.ext.DefaultHandler2;
import ru.yandex.oldwebmaster.compatibility.SimpleXmlBuilder;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.wmtools.common.sita.UserAgentEnum;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * @author aherman
 */
public class DirectOriginalTextService extends AbstractExternalAPIService implements OriginalTextService {
    private static final Logger log = LoggerFactory.getLogger(DirectOriginalTextService.class);

    private static final int DEFAULT_PAGE_SIZE = 10;

    private static final int DEFAULT_MIN_TEXT_LENGTH = 500;
    private static final int DEFAULT_MAX_TEXT_LENGTH = 32000;

    private URI originalsApiUrl;
    private boolean prettyPrint = false;

    private CloseableHttpClient httpClient;
    private int connectionTTLSeconds = 10;
    private int socketTimeoutMillis = 5_000;
    private int connectionTimeoutMillis = HttpConstants.DEFAULT_CONNECT_TIMEOUT;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(connectionTimeoutMillis)
                .setSocketTimeout(socketTimeoutMillis)
                .setRedirectsEnabled(false)
                .build();

        ConnectionConfig connectionConfig = ConnectionConfig.custom()
                .setCharset(StandardCharsets.UTF_8)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(connectionTTLSeconds, TimeUnit.SECONDS)
                .setDefaultConnectionConfig(connectionConfig)
                .setUserAgent(UserAgentEnum.WEBMASTER.getValue())
                .setMaxConnTotal(16)
                .setMaxConnPerRoute(16)
                .build();
    }

    public void destroy() {
        IOUtils.closeQuietly(httpClient);
    }

    private HttpPost createListTextsRequest(WebmasterHostId hostId, int offset, int limit) {
        int pageFrom = getFrom(offset);
        UriComponents requestUri = createRequestUri(hostId, "getReport")
                .queryParam("from", pageFrom)
                .build();
        String requestXml = createListTextsXmlRequest(hostId, pageFrom, limit);
        return createHttpRequest(requestUri, requestXml);
    }

    @ExternalDependencyMethod("list")
    @Override
    public OriginalsResponse listTexts(WebmasterHostId hostId, int offset, int limit) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpPost httpRequest = createListTextsRequest(hostId, offset, limit);
            String response;
            OriginalsResponse originalsResponse;
            try (CloseableHttpResponse httpResponse = queryOriginalsService(httpRequest)) {
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                if (statusCode != HttpStatus.SC_OK) {
                    log.error("Unable to load original texts: host={}, status={}", hostId, statusCode);
                    throw new WebmasterException("Unable to load original texts: host=" + hostId + ", status=" + statusCode,
                            new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), statusCode));
                }
                // В ответе кодировка указана в xml-декларации, но поскольку в парсер прилетает уже строка,
                // он ничего не может с этой информацией сделать. Возможно, лучше протаскивать до парсера InputStream
                response = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new WebmasterException("Unable to load original texts: host=" + hostId,
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }

            try {
                originalsResponse = parseOriginalTextsXml(response);
            } catch (Exception e) {
                throw new WebmasterException("Unable to parse text response",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }

            if (originalsResponse.isErrorState()) {
                throw new WebmasterException("Text loaded with error",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0));
            }
            return originalsResponse;
        });
    }

    static OriginalsResponse parseOriginalTextsXml(String response)
            throws IOException, ParserConfigurationException, SAXException {
        WswDataHandler handler = new WswDataHandler();
        parseContent(response, handler);
        return handler.getResponse();
    }

    private HttpPost createAddTextRequest(WebmasterHostId hostId, String text) {
        UriComponents requestUri = createRequestUri(hostId, "saveData").build();
        String requestXml = createSaveTextXmlRequest(hostId, text);
        return createHttpRequest(requestUri, requestXml);
    }

    @ExternalDependencyMethod("add")
    @Override
    public OriginalsResponse addText(WebmasterHostId hostId, String text) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            String fixedText = text.replaceAll("[\\u0000-\\u0008]|[\\u000B-\\u000C]|[\\u000E-\\u001B]", "");
            HttpPost httpRequest = createAddTextRequest(hostId, fixedText);
            String response;
            try (CloseableHttpResponse httpResponse = queryOriginalsService(httpRequest)) {
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                if (statusCode != HttpStatus.SC_OK) {
                    throw new WebmasterException("Unable to add text",
                            new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), statusCode));
                }
                response = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new WebmasterException("Unable to add text",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }

            OriginalsResponse originalsResponse;
            try {
                // log.debug("Originals response: {}", response);
                originalsResponse = parseOriginalTextsXml(response);
            } catch (Exception e) {
                throw new WebmasterException("Unable parse add text response",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }
            return originalsResponse;
        });
    }

    private HttpPost createGetLimitsRequest(WebmasterHostId hostId) {
        UriComponents requestUri = createRequestUri(hostId, "getSettings").build();
        return createHttpRequest(requestUri, "");
    }

    @ExternalDependencyMethod("limits")
    @Override
    public OriginalTextLimits getLimits(WebmasterHostId hostId) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpPost httpRequest = createGetLimitsRequest(hostId);
            String response;
            try (CloseableHttpResponse httpResponse = queryOriginalsService(httpRequest)) {
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                if (statusCode != HttpStatus.SC_OK) {
                    throw new WebmasterException("Unable to get settings",
                            new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), statusCode));
                }
                response = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new WebmasterException("Unable to get settings",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }

            try {
                return parseLimits(response);
            } catch (Exception e) {
                throw new WebmasterException("Unable to parse settings",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }
        });
    }

    protected OriginalTextLimits parseLimits(String response)
            throws IOException, SAXException, ParserConfigurationException {
        SettingsParser handler = new SettingsParser();
        parseContent(response, handler);
        return handler.getLimits();
    }

    private HttpPost createDeleteTextRequest(WebmasterHostId hostId, String textId) {
        UriComponents requestUri = createRequestUri(hostId, "deleteTask").build();
        String requestXml = createDeleteTextXmlRequest(hostId, textId);
        return createHttpRequest(requestUri, requestXml);
    }

    @ExternalDependencyMethod("delete")
    @Override
    public void deleteText(WebmasterHostId hostId, String textId) {
        trackExecution(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            HttpPost httpRequest = createDeleteTextRequest(hostId, textId);
            try (CloseableHttpResponse httpResponse = queryOriginalsService(httpRequest)) {
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                if (statusCode != HttpStatus.SC_OK) {
                    throw new WebmasterException("Unable to delete original text",
                            new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), statusCode));
                }
                EntityUtils.consume(httpResponse.getEntity());
            } catch (IOException e) {
                throw new WebmasterException("Unable to delete original text",
                        new WebmasterErrorResponse.OriginalsErrorResponse(this.getClass(), 0), e);
            }
        });
    }

    private HttpPost createHttpRequest(UriComponents requestUri, String requestXml) {
        HttpPost post = new HttpPost(requestUri.toUri());
        NameValuePair nameValuePair = new BasicNameValuePair("wsw-fields", requestXml);
        UrlEncodedFormEntity formEntity =
                new UrlEncodedFormEntity(Collections.singletonList(nameValuePair), StandardCharsets.UTF_8);
        post.setEntity(formEntity);
        return post;
    }

    private CloseableHttpResponse queryOriginalsService(HttpPost httpRequest) throws IOException {
        log.info("Query Originals service: {}", httpRequest.getURI());
        CloseableHttpResponse httpResponse = httpClient.execute(httpRequest);
        log.info("Originals response: {} {} {}",
                httpResponse.getStatusLine(),
                httpResponse.getFirstHeader(HttpHeaders.CONTENT_TYPE),
                httpResponse.getFirstHeader(HttpHeaders.CONTENT_LENGTH)
        );
        return httpResponse;
    }

    private static void parseContent(String response, DefaultHandler2 handler)
            throws ParserConfigurationException, SAXException, IOException {
        SAXParserFactory parserFactory = SAXParserFactory.newInstance();
        parserFactory.setNamespaceAware(true);
        parserFactory.setValidating(false);
        parserFactory.setXIncludeAware(false);

        SAXParser parser = parserFactory.newSAXParser();
        parser.parse(new InputSource(new StringReader(response)), handler);
    }

    private UriComponentsBuilder createRequestUri(WebmasterHostId hostId, String operation) {
        return UriComponentsBuilder.fromUri(originalsApiUrl)
                .queryParam("operation", operation)
                .queryParam("hosts", IdUtils.toRobotHostString(hostId))
                .queryParam("host", IdUtils.toRobotHostString(hostId))
                .queryParam("from-service", "WEBMASTER_API");
    }

    private int getFrom(int offset) {
        return offset + 1;
    }

    private String createListTextsXmlRequest(WebmasterHostId hostId, int pageFrom, int pageSize) {
        SimpleXmlBuilder xml = new SimpleXmlBuilder(prettyPrint);
        xml.open("wsw-fields").attribute("pager-from", pageFrom).attribute("pager-items", pageSize);

        xml.open("wsw-field").attribute("name", "host");
        xml.element("wsw-value", IdUtils.toRobotHostString(hostId));
        xml.close(); //wsw-field

        xml.close(); //wsw-fields

        return xml.finish();
    }

    private String createSaveTextXmlRequest(WebmasterHostId hostId, String text) {
        SimpleXmlBuilder xml = new SimpleXmlBuilder(prettyPrint);
        xml.open("wsw-fields").attribute("pager-items", 1);

        xml.open("wsw-field").attribute("name", "host");
        xml.element("wsw-value", IdUtils.toRobotHostString(hostId));
        xml.close(); // wsw-field

        xml.open("wsw-field")
                .attribute("name", "Original_text")
                .attribute("new", true);
        xml.element("wsw-value", text);
        xml.close(); // wsw-field

        xml.close(); // wsw-fields
        return xml.finish();
    }

    private String createDeleteTextXmlRequest(WebmasterHostId hostId, String baseId) {
        SimpleXmlBuilder xml = new SimpleXmlBuilder(prettyPrint);
        xml.open("wsw-fields");

        xml.open("wsw-field").attribute("name", "host");
        xml.element("wsw-value", IdUtils.toRobotHostString(hostId));
        xml.close(); // wsw-field

        xml.open("wsw-field")
                .attribute("name", "Original_text")
                .attribute("deleted", true)
                .attribute("base-id", baseId);
        xml.element("wsw-value", "");
        xml.close(); // wsw-field

        xml.close(); // wsw-fields
        return xml.finish();
    }

    @Required
    public void setOriginalsApiUrl(URI originalsApiUrl) {
        this.originalsApiUrl = originalsApiUrl;
    }

    public void setPrettyPrint(boolean prettyPrint) {
        this.prettyPrint = prettyPrint;
    }

    public void setConnectionTTLSeconds(int connectionTTLSeconds) {
        this.connectionTTLSeconds = connectionTTLSeconds;
    }

    public void setSocketTimeoutMillis(int socketTimeoutMillis) {
        this.socketTimeoutMillis = socketTimeoutMillis;
    }

    public void setConnectionTimeoutMillis(int connectionTimeoutMillis) {
        this.connectionTimeoutMillis = connectionTimeoutMillis;
    }

    private static class WswDataHandler extends DefaultHandler2 {
        private Optional<Boolean> addData = Optional.empty();
        private int totalTexts = 0;

        private List<OriginalText> originalTexts = new ArrayList<>(DEFAULT_PAGE_SIZE);

        private boolean inOriginalTextField = false;
        private String currentBaseId = null;
        private StringBuilder textBuilder = null;
        private Long lastUpdate = null;
        private OriginalsResponse.OriginalsError errorState = OriginalsResponse.OriginalsError.UNKNOWN_ERROR;

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes)
                throws SAXException {
            if ("wsw-error".equals(localName)) {
                String errorCode = attributes.getValue("code");
                if ("text-already-added".equals(errorCode)) {
                    errorState = OriginalsResponse.OriginalsError.TEXT_ALREADY_ADDED;
                } else {
                    errorState = OriginalsResponse.OriginalsError.UNKNOWN_ERROR;
                }
            } else if ("wsw-data".equals(localName)) {
                String addData = attributes.getValue("add-data");
                if (!StringUtils.isEmpty(addData)) {
                    this.addData = Optional.of(Boolean.parseBoolean(addData));
                }
                errorState = OriginalsResponse.OriginalsError.NO_ERROR;
            } else if ("wsw-fields".equals(localName)) {
                String pageName = attributes.getValue("page");
                if (!"Originals-submission-form".equals(pageName)) {
                    return;
                }
                totalTexts = Integer.parseInt(attributes.getValue("pager-count"));
            } else if ("wsw-field".equals(localName)) {
                String pageName = attributes.getValue("name");
                if (!"Original_text".equals(pageName)) {
                    return;
                }

                String baseId = attributes.getValue("base-id");
                if (!StringUtils.isEmpty(baseId)) {
                    currentBaseId = baseId;
                }
            } else if ("wsw-value".equals(localName)) {
                if (currentBaseId != null) {
                    String lastUpdateValue = attributes.getValue("last-update");
                    if (!StringUtils.isEmpty(lastUpdateValue)) {
                        lastUpdate = Longs.tryParse(lastUpdateValue);
                    }
                    textBuilder = new StringBuilder();
                    inOriginalTextField = true;
                }
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if ("wsw-value".equals(localName)) {
                if (inOriginalTextField && textBuilder != null) {
                    String text = textBuilder.toString();
                    DateTime date = null;
                    if (lastUpdate != null) {
                        date = new DateTime(TimeUnit.SECONDS.toMillis(lastUpdate));
                    }

                    originalTexts.add(new OriginalText(currentBaseId, text, date));
                }
                inOriginalTextField = false;
                textBuilder = null;
                lastUpdate = null;
            } else if ("wsw-field".equals(localName)) {
                currentBaseId = null;
            }
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            if (inOriginalTextField) {
                textBuilder.append(ch, start, length);
            }
        }

        public OriginalsResponse getResponse() {
            return new OriginalsResponse(totalTexts, addData, originalTexts, errorState);
        }
    }

    private static class SettingsParser extends DefaultHandler2 {
        private State state = State.UNKNOWN;
        private int minTextLength;
        private int maxTextLength;
        private int textsAdded;
        private int textsCurrentLimit;
        private StringBuilder value;

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws
                SAXException {
            if (!"wsw-value".equals(localName)) {
                return;
            }

            state = State.getState(attributes.getValue("name"));
            if (state != State.UNKNOWN) {
                value = new StringBuilder();
            }
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            if (state == State.UNKNOWN || value == null) {
                return;
            }
            value.append(ch, start, length);
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (state == State.UNKNOWN || value == null) {
                return;
            }
            String valueStr = StringUtils.trimToEmpty(value.toString());
            try {
                switch (state) {
                    case MIN_TEXT_LENGTH:
                        minTextLength = Integer.parseInt(valueStr);
                        break;
                    case MAX_TEXT_LENGTH:
                        maxTextLength = Integer.parseInt(valueStr);
                        break;
                    case TEXTS_ADDED:
                        textsAdded = Integer.parseInt(valueStr);
                        break;
                    case TEXTS_CURRENT_LIMIT:
                        textsCurrentLimit = Integer.parseInt(valueStr);
                        break;
                }
            } catch (NumberFormatException e) {
                log.error("Unable to parse limits: state=" + state + " value=" + valueStr, e);
            }
            value = null;
            state = State.UNKNOWN;
        }

        public OriginalTextLimits getLimits() {
            if (minTextLength < 0 || maxTextLength < 0 || maxTextLength <= minTextLength) {
                return new OriginalTextLimits(DEFAULT_MIN_TEXT_LENGTH, DEFAULT_MAX_TEXT_LENGTH, textsAdded, textsCurrentLimit);
            }
            return new OriginalTextLimits(minTextLength, maxTextLength, textsAdded, textsCurrentLimit);
        }

        private static enum State {
            UNKNOWN,
            MIN_TEXT_LENGTH,
            MAX_TEXT_LENGTH,
            TEXTS_ADDED,
            TEXTS_CURRENT_LIMIT,;

            public static State getState(String name) {
                if (StringUtils.isEmpty(name)) {
                    return UNKNOWN;
                }

                switch (name) {
                    case "min-text-length":
                        return MIN_TEXT_LENGTH;
                    case "max-text-length":
                        return MAX_TEXT_LENGTH;
                    case "texts-added":
                        return TEXTS_ADDED;
                    case "texts-current-limit":
                        return TEXTS_CURRENT_LIMIT;
                }
                return UNKNOWN;
            }
        }
    }
}
