package ru.yandex.webmaster3.storage.payments;

import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.payments.ServiceMerchantInfo;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * Created by Oleg Bazdyrev on 2019-05-28.
 */
public class PaymentsService extends AbstractExternalAPIService {

    private static final Logger log = LoggerFactory.getLogger(PaymentsService.class);

    private static final int socketTimeoutMs = 5000;
    private static final int connectTimeoutMs = HttpConstants.DEFAULT_CONNECT_TIMEOUT;

    private CloseableHttpClient httpClient;

    private final String serviceUrl;
    private final TVMTokenService tvmTokenService;

    @Autowired
    public PaymentsService(String serviceUrl, TVMTokenService tvmTokenService) {
        this.serviceUrl = serviceUrl;
        this.tvmTokenService = tvmTokenService;
    }

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(socketTimeoutMs)
                .setConnectTimeout(connectTimeoutMs)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();
    }

    @ExternalDependencyMethod("create-service-merchant")
    public ServiceMerchantInfo createServiceMerchant(String domain, String token, Long userId) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            try {
                log.info("Creating merchant for domain {} and token {}", domain, token);
                HttpPost post = new HttpPost(serviceUrl + "/internal/service");
                post.addHeader(TVMTokenService.TVM2_TICKET_HEADER, tvmTokenService.getToken());
                ServiceMerchantTokenRequest request = new ServiceMerchantTokenRequest(token, domain,
                        "Turbo-pages merchant for domain " + domain);
                String data = JsonDBMapping.OM.writeValueAsString(request);
                post.setEntity(new StringEntity(data, ContentType.APPLICATION_JSON));
                log.info("Executing request: {} with data {}", post, data);

                try (CloseableHttpResponse response = httpClient.execute(post)) {
                    int code = response.getStatusLine().getStatusCode();
                    String content = EntityUtils.toString(response.getEntity());
                    log.info("Payments service responded with code {} and content {}", code, content);
                    switch (code) {
                        case HttpStatus.SC_OK:
                            ServiceMerchantResponse resp = JsonDBMapping.OM.readValue(content, ServiceMerchantResponse.class);
                            return resp.getData().toServiceMerchantInfo();
                        case HttpStatus.SC_NOT_FOUND:
                            return null;
                        case HttpStatus.SC_CONFLICT:
                        default:
                            throw new WebmasterException("Payments service returns code " + code,
                                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Payments service error"));
                    }
                }
            } catch (IOException e) {
                throw new WebmasterException("Payments service IO exception",
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Payments service error"), e);
            }
        });
    }

    @ExternalDependencyMethod("get-service-merchant")
    public ServiceMerchantInfo getServiceMerchant(long id) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            try {
                log.info("Retrieving merchant for id {}", id);
                HttpGet get = new HttpGet(serviceUrl + "/internal/service/" + id);
                get.addHeader(TVMTokenService.TVM2_TICKET_HEADER, tvmTokenService.getToken());
                log.info("Executing request: {}", get);

                try (CloseableHttpResponse response = httpClient.execute(get)) {
                    int code = response.getStatusLine().getStatusCode();
                    String content = EntityUtils.toString(response.getEntity());
                    log.info("Payments service responded with code {} and content {}", code, content);
                    switch (code) {
                        case HttpStatus.SC_OK:
                            ServiceMerchantResponse resp = JsonDBMapping.OM.readValue(content, ServiceMerchantResponse.class);
                            ServiceMerchantInfo result = resp.getData().toServiceMerchantInfo();
                            return result;
                        case HttpStatus.SC_NOT_FOUND:
                            return null;
                        default:
                            throw new WebmasterException("Payments service returns code " + code,
                                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Payments service error"));
                    }
                }
            } catch (IOException e) {
                throw new WebmasterException("Payments service IO exception",
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Payments service error"), e);
            }
        });
    }

}
