package ru.yandex.webmaster3.storage.payments;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.payments.ServiceMerchantInfo;
import ru.yandex.webmaster3.core.payments.ServiceMerchantOrganization;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * Created by kravchenko99 06.04.21.
 */
@Repository
public class ServiceMerchantCacheYDao extends AbstractYDao {

    private static final DataMapper<ServiceMerchantInfo> MAPPER = DataMapper.create(F.SERVICE_MERCHANT_ID,
            F.SERVICE_ID, F.ACTIVE, F.ENTITY_ID, F.DESCRIPTION, F.ADD_DATE, F.UPDATE_DATE, F.REFRESH_DATE,
            F.ORGANIZATION, ServiceMerchantInfo::new
    );

    private static final DataMapper<Pair<String, String >> PK_MAPPER = DataMapper.create(F.DOMAIN, F.USER_TOKEN, Pair::of);

    protected ServiceMerchantCacheYDao() {
        super(PREFIX_TURBO, "service_merchant_cache");
    }

    public ServiceMerchantInfo findRecord(String domain, String token, Long userId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domain), "domain may not be empty");
        // token may be missing
        token = tokenOrUserId(token, userId);
        return select(MAPPER).where(F.DOMAIN.eq(domain)).and(F.USER_TOKEN.eq(token)).queryOne();
    }

    public void storeRecord(String domain, String token, Long userId, ServiceMerchantInfo merchantInfo) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domain), "domain may not be empty");
        token = tokenOrUserId(token, userId);
        upsert(
                F.DOMAIN.value(domain),
                F.USER_TOKEN.value(token),
                F.SERVICE_MERCHANT_ID.value(merchantInfo.getId()),
                F.USER_ID.value(userId),
                F.SERVICE_ID.value(merchantInfo.getServiceId()),
                F.ACTIVE.value(merchantInfo.isActive()),
                F.ENTITY_ID.value(merchantInfo.getEntityId()),
                F.DESCRIPTION.value(merchantInfo.getDescription()),
                F.ADD_DATE.value(merchantInfo.getAddDate()),
                F.UPDATE_DATE.value(merchantInfo.getLastUpdate()),
                F.REFRESH_DATE.value(new DateTime()),
                F.ORGANIZATION.value(merchantInfo.getOrganization())
        ).execute();
    }

    public void deleteForUser(long userId) {
        var sel = select(PK_MAPPER).secondaryIndex(USER_ID_INDEX).where(F.USER_ID.eq(userId)).getStatement();
        delete().on(sel).execute();
    }

    private String tokenOrUserId(String token, Long userId) {
        if (token == null) {
            Preconditions.checkArgument(userId != null, "Both token and userId may not be empty");
            token = String.valueOf(userId);
        }
        return token;
    }

    private interface F {
        Field<String> DOMAIN = Fields.stringField("domain");
        Field<String> USER_TOKEN = Fields.stringField("user_token");
        Field<Long> SERVICE_MERCHANT_ID = Fields.longField("service_merchant_id");
        Field<Long> USER_ID = Fields.longField("user_id").makeOptional();
        Field<Long> SERVICE_ID = Fields.longField("service_id");
        Field<Boolean> ACTIVE = Fields.boolField("active");
        Field<String> ENTITY_ID = Fields.stringField("entity_id");
        Field<String> DESCRIPTION = Fields.stringField("description");
        Field<DateTime> ADD_DATE = Fields.jodaDateTimeField("add_date");
        Field<DateTime> UPDATE_DATE = Fields.jodaDateTimeField("update_date");
        Field<DateTime> REFRESH_DATE = Fields.jodaDateTimeField("refresh_date").makeOptional();
        Field<ServiceMerchantOrganization> ORGANIZATION = Fields.jsonField2("organization",
                ServiceMerchantOrganization.class);
    }
}
