package ru.yandex.webmaster3.storage.radar.appId;

import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.metrika.radar.HostMetricaAppIdState;
import ru.yandex.webmaster3.core.metrika.radar.MetricaAppIdBindingStateEnum;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

@Repository
public class HostMetricaAppIdYDao extends AbstractYDao {
    private static final String TABLE_NAME = "host_metrika_app_id";

    protected HostMetricaAppIdYDao() {
        super(PREFIX_METRIKA, TABLE_NAME);
    }

    public void add(String hostUrl, long appId, MetricaAppIdBindingStateEnum bindingState,
                    String userLogin, String email, DateTime addTime, DateTime lastUpdate) {
        upsert(
                F.HOST_URL.value(hostUrl),
                F.APP_ID.value(appId),
                F.STATE.value(bindingState),
                F.USER_LOGIN.value(userLogin),
                F.EMAIL.value(email),
                F.ADD_TIME.value(addTime),
                F.LAST_UPDATE.value(lastUpdate)
        )
                .execute();
    }

    public void update(String hostUrl, long appId, MetricaAppIdBindingStateEnum bindingState, DateTime lastUpdate) {
        upsert(
                F.HOST_URL.value(hostUrl),
                F.APP_ID.value(appId),
                F.STATE.value(bindingState),
                F.LAST_UPDATE.value(lastUpdate)
        )
                .execute();
    }

    public List<HostMetricaAppIdState> getAll(String hostUrl) {
        return select(MAPPER)
                .where(F.HOST_URL.eq(hostUrl))
                .queryForList(Pair.of(F.APP_ID, HostMetricaAppIdState::getAppId));
    }

    public MetricaAppIdBindingStateEnum getState(String hostUrl, long appId) {
        return select(F.STATE)
                .where(F.HOST_URL.eq(hostUrl))
                .and(F.APP_ID.eq(appId))
                .queryOne();
    }

    public void delete(long appId, String hostUrl) {
        delete()
                .where(F.APP_ID.eq(appId))
                .and(F.HOST_URL.eq(hostUrl))
                .execute();
    }

    private static final DataMapper<HostMetricaAppIdState> MAPPER = DataMapper.create(
            F.HOST_URL, F.APP_ID, F.STATE,
            F.USER_LOGIN, F.EMAIL, F.ADD_TIME, F.LAST_UPDATE,
            HostMetricaAppIdState::new
    );


    private static class F {
        static final Field<String> HOST_URL = Fields.stringField("host_url");
        static final Field<Long> APP_ID = Fields.longField("app_id");
        static final Field<MetricaAppIdBindingStateEnum> STATE = Fields.stringEnumField("state",
                MetricaAppIdBindingStateEnum.R);
        static final Field<String> USER_LOGIN = Fields.stringField("user_login").makeOptional();
        static final Field<String> EMAIL = Fields.stringField("email").makeOptional();
        static final Field<DateTime> ADD_TIME = Fields.jodaDateTimeField("add_time").makeOptional();
        static final Field<DateTime> LAST_UPDATE = Fields.jodaDateTimeField("last_update").makeOptional();
    }
}
