package ru.yandex.webmaster3.storage.recommendedquery;

import java.util.List;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.recommendedquery.dao.HostsWithOpenedRecommendedYDao;
import ru.yandex.webmaster3.storage.recommendedquery.dao.RecommendedQueriesCHDao;
import ru.yandex.webmaster3.storage.recommendedquery.dao.RecommendedQueriesLastVisitDateYDao;
import ru.yandex.webmaster3.storage.recommendedquery.filtres.RecommendedQueryField;
import ru.yandex.webmaster3.storage.recommendedquery.samples.RecommendedQuery;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;

/**
 * Created by Oleg Bazdyrev on 11/03/2021.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RecommendedQueriesService implements UserTakeoutDataProvider {

    private final RecommendedQueriesCHDao mdbRecommendedQueriesCHDao;
    private final RecommendedQueriesLastVisitDateYDao recommendedQueriesLastVisitDateYDao;
    private final HostsWithOpenedRecommendedYDao hostsWithOpenedRecommendedYDao;

    public long getQueriesCount(WebmasterHostId hostId, Condition filters, RecommendedQueriesCHDao.Include include) {
        return mdbRecommendedQueriesCHDao.getQueriesCount(hostId, filters, include);
    }

    public boolean hasQueries(WebmasterHostId hostId, RecommendedQueriesCHDao.Include include) {
        return mdbRecommendedQueriesCHDao.exist(hostId, RecommendedQueriesCHDao.Include.BOTH);
    }

    public List<RecommendedQuery> getQueries(
            WebmasterHostId hostId, Condition filters, RecommendedQueryField orderBy,
            OrderBy.Direction orderDirection, long limitFrom, long limitSize, RecommendedQueriesCHDao.Include include) {
        return mdbRecommendedQueriesCHDao.getQueries(hostId, filters, orderBy, orderDirection, limitFrom, limitSize, include);
    }

    public long getRegionsCount(WebmasterHostId hostId, RecommendedQueriesCHDao.Include include) {
        return mdbRecommendedQueriesCHDao.getRegionsCount(hostId, include);
    }

    public List<Long> getRegions(WebmasterHostId hostId, long limitFrom, long limitSize, RecommendedQueriesCHDao.Include include) {
        return mdbRecommendedQueriesCHDao.getRegions(hostId, limitFrom, limitSize, include);
    }

    public void updateLastVisitDate(WebmasterHostId hostId) {
        recommendedQueriesLastVisitDateYDao.insert(hostId, LocalDate.now());
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        hostsWithOpenedRecommendedYDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                hostsWithOpenedRecommendedYDao.getTablePath()
        );
    }
}
